% compactness() - evaluate compactness of component ROA map by
%                 using pair distance measurement (pdist() provided
%                 by matlab.
%
% Usage:
%    >> normr = compactness(component_no,z_threshold,...
%                  dist_method);
%
% Inputs:
%    component_no - component number
%    z_threshold  - z value threshold for ROA maps
%    dist_method  - method to derive pair distance, options:
%                 - 'euclid'
%                 - 'seuclid'
%                 - 'cityblock'
%                 - 'mahal'
%                 - 'minkowski'
%
% output:
%    normr - norm of the residuals when the cdf of the histogram of
%            cluster distribution is fitted by the polynomial
%            (degree of 2)
%
% Author: Jeng-Ren Duann, SCCN/INC/UCSD & CNL/Salk Inst., 2003-02-19
%
% See also: pdist(), gamfit(), gampdf()
%
% Note: 
%    - This function need to be launched under fmrlab with fmr
%    dataset loaded.
%    - pdist(), gamfit() and gampdf() functions need
%    matlab/statistics toolbox

% --
% Copyright (C) 2002 Jeng-Ren Duann, Salk Institute, duann@salk.edu
%
% This program is free software; you can redistribute it and/or
% modify it under the terms of the GNU General Public License as
% published by the Free Software Foundation; either version 2 of
% the License, or (at your option) any later version.
%
% This program is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU General Public License for more details.
%
% You should have received a copy of the GNU General Public License
% along with this program; if not, write to the Free Software
% Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

% $Log: compactness.m,v $
% Revision 1.1  2003/02/06 19:09:56  duann
% Initial revision

function normr=compactness(component_no,z_threshold,spatial_ext)  
  
if nargin < 3,
  spatial_ext=3;
end
if nargin < 2,
  z_threshold = 1.5;
end

global FMRI

if isunix,
    text_size = 14;
else
    text_size = 12;
end

W = FMRI.icaweight * FMRI.icasphere;
u = W * FMRI.data;

activation = u(component_no,:);
activation = (activation - mean(activation)) / std(activation);
ii = find(abs(activation) < z_threshold);
activation(ii) = 0;
ii = find(abs(activation) >= z_threshold);
activation(ii) = 1;
whole_map = zeros(1,FMRI.dime_x*FMRI.dime_y*FMRI.dime_z);
for i=1:length(FMRI.idx),
  whole_map(FMRI.idx(i)) = activation(i);
end
whole_map = reshape(whole_map,FMRI.dime_x,FMRI.dime_y,FMRI.dime_z);

conn3 = conndef(3,'min');
c = convn(whole_map,conn3,'same');
conn = c;
ii = find(c == 1);
c(ii) = 0;
ii = find(c ~= 0);
c(ii) = fmrlab_scale(c(ii),104,128);
c = reshape(c,FMRI.dime_x,FMRI.dime_y,FMRI.dime_z);
temp = mean(FMRI.data);
str = zeros(1,FMRI.dime_x*FMRI.dime_y*FMRI.dime_z);
for i=1:length(FMRI.idx),
  str(FMRI.idx(i)) = temp(i);
end
str = fmrlab_scale(str,1,64);
str = reshape(str,FMRI.dime_x,FMRI.dime_y,FMRI.dime_z);
c = max(c,str);
c = reshape(c,FMRI.dime_x,FMRI.dime_y,1,FMRI.dime_z);
for i=1:FMRI.dime_z;
  temp = c(:,:,1,i);
  c(:,:,1,i) = flipud(temp);
end
fig = figure;
pos = get(fig,'position');
pos(4) = 600;
set(fig,...
    'numbertitle','off',...
    'name','Compactness Measurement',...
    'position',pos);
ti = textsc(0.5,0.95,['Component #' num2str(component_no)]);
set(ti,'fontsize',text_size+2);
sbplot(4,1,[1 3]);
montage(c,fmrlab_color(2));

% convert conn to binary and use bwlabeln() to label the groups
conn = conn(:);
ii = find(conn ~=0);
bin_conn = zeros(size(conn));
bin_conn(ii) = 1;
bin_conn = reshape(bin_conn,FMRI.dime_x,FMRI.dime_y,FMRI.dime_z);
l_bin_conn = bwlabeln(bin_conn,18);
ii = find(l_bin_conn ~= 0);
sbplot(4,1,4);
[n,x]=hist(l_bin_conn(ii),floor(sqrt(length(l_bin_conn(ii)))));
%[phat,pci]=gamfit(n);
%y = gampdf(x,phat(1),phat(2));
n = fliplr(sort(n));
bar(x,n);
hold on
%plot(x,y*max(n)/max(y),'b');

ax1 = gca;
pos = get(ax1,'position');
ax2 = axes('position',pos,...
	   'xaxislocation','top',...
	   'yaxislocation','right',...
	   'color','none',...
	   'xcolor','r',...
	   'ycolor','r');
y(1) = n(1);
for i=2:length(x),
  y(i) = y(i-1)+n(i);
end
y = y/sum(n);
line(x,y,'parent',ax2,'color','r');
[p,s]=polyfit(x,y,1);
normr = s.normr;
y1 = polyval(p,x);
line(x,y1,'color','b','parent',ax2);
str = sprintf('%6.4fx+%6.4f        R=%6.4f',...
	      p(1),p(2),normr);
a = axis;
tx = text(0.1*a(2),0.9*a(4)+0.1*a(3),str);
set(tx,'fontsize',text_size-4,'color','b');

%l_bin_conn(ii) = l_bin_conn(ii) + 80;
%l_bin_conn = reshape(l_bin_conn,FMRI.dime_x,FMRI.dime_y,1, ...
%		     FMRI.dime_z);
%for i=1:FMRI.dime_z;
%  temp = l_bin_conn(:,:,1,i);
%  l_bin_conn(:,:,1,i) = flipud(temp);
%end
%sbplot(3,4,[3 8]);
%montage(l_bin_conn,fmrlab_color(2));

return;

max_loc = find(conn==max(conn(:)));
max_loc_x = mod(max_loc,FMRI.dime_x);
max_loc_y = mod(floor(max_loc/FMRI.dime_x),FMRI.dime_y);
max_loc_z = 1+floor(max_loc/FMRI.dime_x/FMRI.dime_y);

% find the coordinates of voxels which are with activation > z threshold
ii = find(abs(activation) >= z_threshold);

x = mod(FMRI.idx(ii),FMRI.dime_x);
y = mod(floor(FMRI.idx(ii)/FMRI.dime_x),FMRI.dime_y);
z = 1+floor(FMRI.idx(ii)/FMRI.dime_x/FMRI.dime_y);

d = [x; y; z]';  
dd = pdist(d,'cityblock');

mdist = zeros(length(ii),length(ii));
k=1;
for i=1:length(ii)-1,
  for j=(i+1):length(ii),
    mdist(i,j) = dd(k);
    k = k+1;
  end
end

[i,j]=find(mdist>0 & mdist<=2);
whos i j
figure
plot3(x(i),y(i),max(z)-z(i),'r*');

mdist = mdist + mdist';
min_dist = zeros(1,length(ii));
for i=1:length(ii),
  temp = mdist(i,:);
  temp(i) =[];
  min_dist(i) = min(temp);
end

% create the gamma fit to evaluate the distribution of compactness
[phat,pci]=gamfit(min_dist);
[n,x]=hist(min_dist,100);
n = n/length(min_dist); % normalize the distribution
deltax = min(x):(max(x)-min(x))/100:max(x);
y = gampdf(deltax,phat(1),phat(2));
figure
bar(x,n);
hold on
plot(deltax,y*max(n)/max(y),'r');
title(['Component ' num2str(component_no)]);
str = sprintf('gamma(%5.3f, %5.3f)',phat(1),phat(2));
a = axis;
tx = text(0.5*(a(2)+a(1)),0.9*(a(4)+a(3)),str);
set(tx,'fontsize',text_size,'color','r');
