% fmrimovie() - make fmri movies by using the component map and time course
%               obtained in ICA analysis (put all components together)
%
% Usage:
%    >> [Movie, Map] = fmrimovie3();
%
% Inputs:
%
% Outputs:
%    Movie - matrix of fMRI movies
%    Map - colormap used in the fMRI movie
%
% Author: Jeng-Ren Duann, INC/UCSD and CNL/Salk, 2002-12-02
%
% See also: fmrlab_color()

% Copyright (C) 2002 Jeng-Ren Duann, Salk Institute, duann@salk.edu
%
% This program is free software; you can redistribute it and/or
% modify it under the terms of the GNU General Public License as
% published by the Free Software Foundation; either version 2 of
% the License, or (at your option) any later version.
%
% This program is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU General Public License for more details.
%
% You should have received a copy of the GNU General Public License
% along with this program; if not, write to the Free Software
% Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

% $Log: fmrimovie.m,v $
% Revision 1.1  2003/02/06 18:59:13  duann
% Initial revision
%

function fmrimovie()


  global FMRI
  
  RGB_mainwin = [1 .984 .895];
  RGB_background = [0.860 0.824 0.618];
  RGB_button = [0.860 0.824 0.618];
  RGB_text = [0.333 0.283 0.003];
  
% 1. popup an input dialogue for users to input:
%    components to be processed (up to 6):
%    z-value threshold
%    skip frames
%    target movie file name
prompt = {'Components to make movies (up to 6):',...
	  'Z-value threshold:',...
	  'How many frames to skip between two successive frames:',...
	  'File name of Movie (*.mpg):'};
defs = {'',...
	'2.0',...
	'2',...
	'test.mpg'};
a = fmrlab_inputdlg(prompt, 'FMRI Movie',1,defs);
if isempty(a),
  %errordlg('Cancelled by user...');
  %fmrlab_messagedlg('FMRI Movie',...
%		'Warning:',...
%		'Cancelled by user');
  return;
end
components = str2num(char(a(1)));
if isempty(components),
  %errordlg('No components specified...!');
  fmrlab_messagedlg('FMRI Movie',...
		'Error:',...
		'No components specified');
  return;
end
z_thre = str2num(char(a(2)));
skip = str2num(char(a(3)));
movie_name = char(a(4));

% 2. make available base image
%    structural image exists -> base_image = structural
%    structural image doesn't exist -> base_image =
%    sqrt(std(FMRI.data))
if isempty(FMRI.struc_name),
  temp = sqrt(std(FMRI.data));
  base_image = zeros(1,FMRI.dime_x*FMRI.dime_y*FMRI.dime_z);
  for i=1:length(FMRI.idx),
    base_image(FMRI.idx(i)) = temp(i);
  end
else
  fname = [FMRI.struc_path FMRI.struc_name];
  fp = fopen(fname,'rb','ieee-be');
  base_image = fread(fp,'uint16')';
  fclose(fp);
end

% 3. scale base image to range [1:64]
base_image = (base_image-min(base_image)) * 64 / ...
    (max(base_image)-min(base_image));

% 4. make a figure to display base_image first
pos = get(0,'screensize');
h = figure(...
    'numbertitle','off',...
    'menubar','none',...
    'color',RGB_mainwin,...
    'name','fMRI Movie',...
    'position',[(pos(3)-640)/2 (pos(4)-640)/2 640 640]);    
disp_col = ceil(sqrt(FMRI.dime_z));
disp_row = ceil(FMRI.dime_z/disp_col);
if mod(FMRI.dime_z,disp_col) == 0,   
  disp_row = disp_row + 1;
end
disp_row = disp_row + 1;
if isempty(FMRI.struc_name),
  temp = reshape(base_image,...
		 FMRI.dime_x,...
		 FMRI.dime_y,...
		 FMRI.dime_z);
else
  temp = reshape(base_image,...
		 FMRI.struc_image_width,...
		 FMRI.struc_image_height,...
		 FMRI.struc_image_depth);
end
for i=1:FMRI.dime_z,
  a = floor(i/disp_col);
  b = mod(i,disp_col);
  tightsubplot(disp_row,disp_col,a*disp_col+b);
  imagesc(temp(:,:,i)');
  caxis([1 128]);
  axis off
  axis image
end
fmrlab_color(4,length(components));


map = fmrlab_color(4,length(components));
map(1:64,:) = [];
gap = fix(64/length(components));
for i=1:length(components),
  a = tightsubplot(disp_row,disp_col*length(components),...
	       disp_row*disp_col*length(components)- ...
	       length(components)+i);
  axis off
  pos = get(a,'position');
  set(a,'position',[pos(1) pos(2)-0.15*pos(4) pos(3) pos(4)]);
  mm = map(1+(i-1)*gap:i*gap,:);
  mm = reshape(mm,gap,1,3);
  image(mm(end:-1:1,:,:));
  axis off
  axis image
end

% 5. calculate the component maps
W = FMRI.icaweight * FMRI.icasphere;
invW = pinv(W);
u = W * FMRI.data;

% 6. making frames
% remove means of time course of every component
activation = rmbase(invW(:,components)');
tc_panel = tightsubplot(disp_row,disp_col,...
           [(disp_row-1)*disp_col+1 disp_col*disp_row-disp_col/3]);
pos = get(tc_panel,'position');
set(tc_panel,'position',[pos(1) pos(2)-0.15*pos(4) pos(3) pos(4)]);
eegplotsold(activation, 1/FMRI.tr, 0, 'fMRI Time Courses', ...
	   10,FMRI.timepoint*FMRI.tr,'b');

ii = 1;
for tp = 1:skip:size(invW,1),
  disp(['Time point #' num2str(tp) ' is processed...']);
  % determine displayed value for each component
  value = zeros(1,length(components));
  for i=1:length(components)
    ubound = max(activation(i,:));
    lbound = min(activation(i,:));
    value(i) = (activation(i,tp)-lbound) * ...
	fix(64/length(components)) / ...
	(ubound - lbound);
  end
  % sort value to determine the sequence of components
  % to make movies (ascending)
  [a, comp_idx] = sort(value);
  % make map for each component
  for i=1:length(comp_idx),
    temp = u(components(comp_idx(i)),:);
    temp = (temp-mean(temp))/std(temp);
    uidx = find(temp >= z_thre);
    umap = zeros(1,length(temp));
    umap(uidx) = value(comp_idx(i)) + ...
	(65 + (comp_idx(i)-1)*fix(64/length(components)));
    uimg = zeros(1,FMRI.dime_x*FMRI.dime_y*FMRI.dime_z);
    for j=1:length(FMRI.idx),
      uimg(FMRI.idx(j)) = umap(j);
    end
    
    % interpolating activation map if necessary
    if ~isempty(FMRI.struc_name),
      uimg = reshape(uimg,...
		     FMRI.dime_x,...
		     FMRI.dime_y,...
		     FMRI.dime_z);
      xs = linspace(1,FMRI.struc_image_width,FMRI.dime_x);
      ys = linspace(1,FMRI.struc_image_height,FMRI.dime_y);
      [xi,yi] = meshgrid(1:FMRI.struc_image_width,...
			 1:FMRI.struc_image_height);
      smap = [];
      for j=1:FMRI.struc_image_depth,
	temp = interp2(xs,ys,uimg(:,:,j),xi,yi);
	temp = reshape(temp,...
		       FMRI.struc_image_width*...
		       FMRI.struc_image_height,1);
	smap = [smap temp];
      end
      uimg = reshape(smap,1,...
		     FMRI.struc_image_width*...
		     FMRI.struc_image_height*...
		     FMRI.struc_image_depth);
    end
    
    % combining uimg and base image
    uidx = find(uimg < (65+(comp_idx(i)-1)*...
			fix(64/length(components))));
    uimg(uidx) = 0;
    uidx = find(uimg >= (65+(comp_idx(i)-1)*...
			fix(64/length(components))));
    if i==1,
      img = base_image;
    end
    img(uidx) = 0;
    img = img + uimg;
  end
  
  figure(h);
  if isempty(FMRI.struc_name),
    img = reshape(img,...
		  FMRI.dime_x,...
		  FMRI.dime_y,...
		  FMRI.dime_z);
  else
    img = reshape(img,...
		  FMRI.struc_image_width,...
		  FMRI.struc_image_height,...
		  FMRI.struc_image_depth);
  end
  for i=1:FMRI.dime_z,
    a = floor(i/disp_col);
    b = mod(i,disp_col);
    tightsubplot(disp_row,disp_col,a*disp_col+b);
    imagesc(img(:,:,i)');
    caxis([1 128]);
    axis off
    axis image
  end
  fmrlab_color(4,length(components));
  
  % plot component time courses with vertical line
  % indicating the time of currence
  tc_panel = tightsubplot(disp_row,disp_col,...
			  [(disp_row-1)*disp_col+1 disp_col*disp_row-disp_col/3]);
  pos = get(tc_panel,'position');
  set(tc_panel,'position',[pos(1) pos(2)-0.15*pos(4) pos(3) pos(4)]);
  eegplotsold(activation, 1/FMRI.tr, 0, 'fMRI Time Courses', ...
	   10,FMRI.timepoint*FMRI.tr,'b');
  ll = line([(tp-1)*FMRI.tr (tp-1)*FMRI.tr],[0 100]);
  set(ll,'color','r');
  
  % drawing arrows to indicate the strength of activations
  gap = fix(64/length(components));
  for i=1:length(components),
    a = tightsubplot(disp_row,disp_col*length(components),...
	       disp_row*disp_col*length(components)- ...
	       length(components)+i);
    axis off
    pos = get(a,'position');
    set(a,'position',[pos(1) pos(2)-0.15*pos(4) pos(3) pos(4)]);
    mm = map(1+(i-1)*gap:i*gap,:);
    mm = reshape(mm,gap,1,3);
    image(mm(end:-1:1,:,:));
    axis off
    axis image
    
    ll = line([0 2],[gap-value(i) gap-value(i)]);
    set(ll,'color','w');
  end
  
  tx = textsc(0.5, 0.95, 'Region of Activity');
  if isunix,
      text_size = 14;
  else
      text_size = 12;
  end
  set(tx,'fontsize',text_size);
  
  fname = sprintf('temp%04d.fig',ii);
  ii = ii + 1;
  saveas(gcf,fname,'fig');
end

makemovie({'temp' 1 ii-1 4},'outsize',[640 640],'shape','stretch');
