% fmrlab_render() - overlay the ICA ROA maps onto the 3D standard brain
%               template.
%
% Usage: 
%   >> fmrlab_render(dat, brt, rendfile);
%
% Inputs:
%   dat      - data structure holds the information for 3D
%              coordinates and Z values for overlaying
%   brt      - display with opaque color blobs, nan for displaying
%              the results in OLD fasion; 0.25 for lots enhancement; 0.5 for
%              more enhancement; 0.75 for slight enhancement; and 1 for non
%              enhancement
%   rendfile - 3D rendered standard brain template from SPM 99
%              (http://fil.ion.ucl.ac.uk/spm), the default is
%              'render_spm96.mat'
% 
% See also: fmrlab_normalize(), spm_render()
%
% Notes: - This function is adapted from SPM render routines.
%        - 'render_signle_subj.mat' and 'render_smooth_average.mat'
%           are also available
%        - The results are highly dependent on the quality of
%          spatial normalization on both structural and function (or
%          ICA ROA) images. Any misalignment will make the resulting
%          3D rendering look abnormal (e.g., painting out of
%          brain).
%

% Copyright (C) 2002 Jeng-Ren Duann, Salk Institute, duann@salk.edu
%
% This program is free software; you can redistribute it and/or
% modify it under the terms of the GNU General Public License as
% published by the Free Software Foundation; either version 2 of
% the License, or (at your option) any later version.
%
% This program is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU General Public License for more details.
%
% You should have received a copy of the GNU General Public License
% along with this program; if not, write to the Free Software
% Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

% $Log: fmrlab_render.m,v $
% Revision 1.1  2003/02/06 19:00:50  duann
% Initial revision
%

function fmrlab_render(dat,brt,rendfile)
% Adapted from spm_render.m to overlay the ICA results onto the 3D standard
% 	brain template.
%       brt = NaN for displaying the result in OLD fashion
%       brt = 0.25, 0.5, 0.75, and 1 for NEW fasion display with obaque color 
%             blobs. 0.25 -> lots enhancement; 0.5 -> more enhancement; 0.75 ->
%             slight enhancement; and 1 -> none enhancement
%
% Modification:
%       CODE: t0  = t(msk).*exp((log(0.5)/10)*dst)'; is changed to 
%       CODE: t0  = t(msk).*exp((log(0.5)/10)*dst); ref to line 168
%          
% 2001-08-31, Jeng-Ren Duann, CNL/Salk Inst.
% 2002-03-26, Make program independent to SPM environment -JR-
% 2002-03-26, add round funtion in sparse function to get rid of
%             non-integer error, -JR-
  
% Perform the rendering
%=======================================================================
load(rendfile);

if (exist('rend') ~= 1), % Assume old format...
	rend = cell(size(Matrixes,1),1);
	for i=1:size(Matrixes,1),
		rend{i}=struct('M',eval(Matrixes(i,:)),...
			'ren',eval(Rens(i,:)),...
			'dep',eval(Depths(i,:)));
		rend{i}.ren = rend{i}.ren/max(max(rend{i}.ren));
	end;
end;

for i=1:length(rend),
	rend{i}.max=0;
	rend{i}.data = cell(size(dat,1),1);
	if issparse(rend{i}.ren),
		% Assume that images have been DCT compressed
		% - the SPM99 distribution was originally too big.
		d = size(rend{i}.ren);
		B1 = spm_dctmtx(d(1),d(1));
		B2 = spm_dctmtx(d(2),d(2));
		rend{i}.ren = B1*rend{i}.ren*B2';
		% the depths did not compress so well with
		% a straight DCT - therefore it was modified slightly
		rend{i}.dep = exp(B1*rend{i}.dep*B2')-1;
	end;
	msk = find(rend{i}.ren>1);rend{i}.ren(msk)=1;
	msk = find(rend{i}.ren<0);rend{i}.ren(msk)=0;
end;

mx = zeros(length(rend),1)+eps;
for j=1:length(dat),
	XYZ = dat(j).XYZ;
	t   = dat(j).t;
	dim = dat(j).dim;
	mat = dat(j).mat;

	for i=1:length(rend),

		% transform from Taliarach space to space of the rendered image
		%-------------------------------------------------------
		M1  = rend{i}.M*dat(j).mat;
		zm  = sum(M1(1:2,1:3).^2,2).^(-1/2);
		M2  = diag([zm' 1 1]);
		M  = M2*M1;
		cor = [1 1 1 ; dim(1) 1 1 ; 1 dim(2) 1; dim(1) dim(2) 1 ;
		       1 1 dim(3) ; dim(1) 1 dim(3) ; 1 dim(2) dim(3); dim(1) dim(2) dim(3)]';
		tcor= M(1:3,1:3)*cor + M(1:3,4)*ones(1,8);
		off = min(tcor(1:2,:)');
		M2  = spm_matrix(-off+1)*M2;
		M  = M2*M1;
		xyz = (M(1:3,1:3)*XYZ + M(1:3,4)*ones(1,size(XYZ,2)));
		d2  = ceil(max(xyz(1:2,:)'));

		% calculate 'depth' of values
		%-------------------------------------------------------
		dep = spm_slice_vol(rend{i}.dep,spm_matrix([0 0 1])*inv(M2),d2,1);
		z1  = dep(round(xyz(1,:))+round(xyz(2,:)-1)*size(dep,1));

		if ~finite(brt), 
		  msk = find(xyz(3,:) < (z1+20) & xyz(3,:) > (z1-5));
		else,      
		  msk = find(xyz(3,:) < (z1+60) & xyz(3,:) > (z1-5));
		end;

		if ~isempty(msk),

			% generate an image of the integral of the blob values.
			%-----------------------------------------------
			xyz = xyz(:,msk);
			if ~finite(brt), 
			  t0  = t(msk);
			else,	
			  dst = xyz(3,:) - z1(msk);
			  dst = max(dst,0);
			  t0  = t(msk).*exp((log(0.5)/10)*dst);
			end;
			X0  = full(sparse(round(xyz(1,:)), round(xyz(2,:)), t0, d2(1), d2(2)));
			hld = 1; if ~finite(brt), hld = 0; end;
			X   = spm_slice_vol(X0,spm_matrix([0 0 1])*M2,size(rend{i}.dep),hld);
			msk = find(X<0);
			X(msk) = 0;
		else,
			X = zeros(size(rend{i}.dep));
		end;

		% Brighten the blobs
		if isfinite(brt), 
		  X = X.^brt; 
		end;
		mx(j) = max([mx(j) max(max(X))]);
		rend{i}.data{j} = X;
	end;
end;

mxmx = max(mx);

nrow = ceil(length(rend)/2);
h = figure(...
    'menubar','none',...
    'toolbar','none',...
    'numbertitle','off');
ax=axes('Parent',h,'units','normalized','Position',[0, 0, 1, 1],'Visible','off');
image(0,'Parent',ax);
colormap(gray);
set(ax,'YTick',[],'XTick',[]);

if ~finite(brt),
	% Old style split colourmap display.
	%---------------------------------------------------------------
	load Split;
	colormap(split);
	for i=1:length(rend),
		ren = rend{i}.ren;
		X   = rend{i}.data{1}/mxmx;
		msk = find(X);
		ren(msk) = X(msk)+1;
		ax=axes('Parent',h,'units','normalized',...
			'Position',[rem(i-1,2)*0.5, floor((i-1)/2)/nrow, 0.5, 1/nrow],...
			'Visible','off');
		image(ren*64,'Parent',ax);
		set(ax,'DataAspectRatio',[1 1 1], ...
			'PlotBoxAspectRatioMode','auto',...
			'YTick',[],'XTick',[],'XDir','normal','YDir','normal');
	end;
else,
	% Combine the brain surface renderings with the blobs, and display using
	% 24 bit colour.
	%---------------------------------------------------------------
	for i=1:length(rend),
		ren = rend{i}.ren;
		X = cell(3,1);
		for j=1:length(rend{i}.data),
			X{j} = rend{i}.data{j}/mxmx;
		end
		for j=(length(rend{i}.data)+1):3
			X{j}=zeros(size(X{1}));
		end

		rgb = zeros([size(ren) 3]);
		tmp = ren.*max(1-X{1}-X{2}-X{3},0);
		rgb(:,:,1) = tmp + X{1};
		rgb(:,:,2) = tmp + X{2};
		rgb(:,:,3) = tmp + X{3};

		ax=axes('Parent',h,'units','normalized',...
			'Position',[rem(i-1,2)*0.5, floor((i-1)/2)/nrow, 0.5, 1/nrow],...
			'Visible','off');
		image(rgb,'Parent',ax);
		set(ax,'DataAspectRatio',[1 1 1], ...
			'PlotBoxAspectRatioMode','auto',...
			'YTick',[],'XTick',[],...
			'XDir','normal','YDir','normal');
	end;
end;

return;

