% map_on_fmri2() - component browser by overlaying ROA onto 2D
%                 slices of functional images with interactive 
%                 graphic user interface
%
% Usage:
%   >> map_on_fmri2(action)
%
% Inputs:
%   action - specifying action for the browse window to perform,
%            the value can be:
%          - 'init': constructing all elements in GUI and linking all events
%          - 'backward': switch the currently displayed component to the 
%                        previous one
%          - 'foreward': switch the currently displayed component to the 
%                        next one
%          - 'refresh': redraw the entire component browser (including the 
%                       time course and bold image plot window, if applicable)
%          - 'goto_select_list': switch the currently displayed component to 
%                                the one indicated in select-component list
%          - 'add_select': add the currently displayed component to select-
%                          component list
%          - 'remove_select': remove the currently displayed component from 
%                             the select-component list indicated 
%          - 'goto_reject': switch the currently displayed component to the one
%                           indicated in the reject-component list
%          - 'add_reject': add the currently displayed component to reject-
%                          component list
%          - 'remove_reject': remove the currently displayed component from
%                             the reject-component list
%          - 'op_clicked': 'operations' buttons selected
%          - 'pgdn': page down button pressed.
%          - 'pgup': page up button pressed.
%
% Outputs:
%
% Author: Jeng-Ren Duann, SCCN/INC/UCSD & CNL/Salk Inst., 2003
%
% See also: map_on_struc2(), boldimage(), cbar()
%
% Notes: - map_on_struc() might need interpolation to make the ROA
%          the same coordinates with structural images. Thus, it might take
%          longer to browse component maps. It's highly recommended to use
%          map_on_fmri() first to add component of interest into the
%          selection list, then to use map_on_struc() to browse the selected
%          component only

% Copyright (C) 2003 Jeng-Ren Duann, SCCN/INC/UCSD, duann@sccn.ucsd.edu
%
% This program is free software; you can redistribute it and/or
% modify it under the terms of the GNU General Public License as
% published by the Free Software Foundation; either version 2 of
% the License, or (at your option) any later version.
%
% This program is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU General Public License for more details.
%
% You should have received a copy of the GNU General Public License
% along with this program; if not, write to the Free Software
% Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

% $Log: map_on_fmri2.m,v $
%

function map_on_fmri2(action)

global FMRI

if length(FMRI) == 0,
    fmrlab_messagedlg('map_on_fmri2()',...
	    'Error:',...
		['No current data set exists. Please load a ''.fmr''' ...
		' data file first.']);
    return;
end

if isempty(FMRI.icaweight),
    fmrlab_messagedlg('map_on_fmri2()',...
        'Error:',...
        'No ICA results available, please run ICA first');
    return;
end

if nargin < 1,
  action = 'init';
end
  
RGB_mainwin = [1 .984 .895];
RGB_background = [0.860 0.824 0.618];
RGB_button = [0.860 0.824 0.618]; 
RGB_text = [0.333 0.283 0.003];

if isunix,
  text_size = 14;
else
  text_size = 12;
end

switch(action)
case 'init'
    f_main = findobj('name','Display Component ROA on Functional Variance Images');
    if isempty(f_main),
        f_main_pos = [5 260 735 480];
        f_main = figure(...
  	        'name', 'Display Component ROA on Functional Variance Images', ...
   	        'numbertitle','off',...
       	    'doublebuffer', 'on', ...
   	        'color',RGB_mainwin,...
   	        'menubar','none',...
   	        'position',f_main_pos,...
   	        'backingstore','off',...
   	        'deletefcn','map_on_fmri2(''close'');');
    
        % create title bar
        pos = [1 f_main_pos(4)-20 f_main_pos(3) 20];
        h1 = uicontrol(...
            'style','text',...
            'Parent',f_main, ...
            'Units','pixels', ...
            'BackgroundColor',RGB_mainwin, ...
            'FontSize',text_size, ...
            'ForegroundColor',RGB_text, ...
            'Position',pos, ...
            'String','Independent Component #1', ...
            'Tag','title_string');

        % create buttons
        %   name of button group 1
        pos = [5 30 80 20];
        h1 = uicontrol(...
            'style','text',...
            'Parent',f_main, ...
            'Units','pixels', ...
            'BackgroundColor',RGB_mainwin, ...
            'FontSize',text_size, ...
            'ForegroundColor',RGB_text, ...
            'Position',pos, ...
            'horizontalalign','left',...
            'String','Browse');
        %   backword button
        pos = [5 5 25 25];
        h1 = uicontrol(...
            'style','pushbutton',...
            'Parent',f_main, ...
            'Units','pixels', ...
            'BackgroundColor',RGB_button, ...
            'FontSize',text_size, ...
            'ForegroundColor',RGB_text, ...
            'Position',pos, ...
            'String','<', ...
            'Tag','but_back',...
            'callback','map_on_fmri2(''backward'');');
        %   foreward button
        pos = [35 5 25 25];
        h1 = uicontrol(...
            'style','pushbutton',...
            'Parent',f_main, ...
            'Units','pixels', ...
            'BackgroundColor',RGB_button, ...
            'FontSize',text_size, ...
            'ForegroundColor',RGB_text, ...
            'Position',pos, ...
            'String','>', ...
            'Tag','but_fore',...
            'callback','map_on_fmri2(''foreward'');');
        %   indicater 1
        pos = [65 5 40 25];
        h1 = uicontrol(...
            'style','edit',...
            'Parent',f_main, ...
            'Units','pixels', ...
            'BackgroundColor',RGB_mainwin, ...
            'FontSize',text_size, ...
            'ForegroundColor',RGB_text, ...
            'Position',pos, ...
            'String','1', ...
            'Tag','ed_component',...
            'callback','map_on_fmri2(''refresh'');');
    
        % name of button group 2 Select Componnet
        pos = [110 30 150 20];
        h1 = uicontrol(...
            'style','text',...
            'Parent',f_main, ...
            'Units','pixels', ...
            'BackgroundColor',RGB_mainwin, ...
            'FontSize',text_size, ...
            'ForegroundColor',RGB_text, ...
            'Position',pos, ...
            'horizontalalign','left',...
            'String','Select Components');
        % mark button for select component
        pos = [110 5 45 25];
        h1 = uicontrol(...
            'style','pushbutton',...
            'Parent',f_main, ...
            'Units','pixels', ...
            'BackgroundColor',RGB_button, ...
            'FontSize',text_size, ...
            'ForegroundColor',RGB_text, ...
            'Position',pos, ...
            'String','Mark', ...
            'Tag','but_mark_select',...
            'callback','map_on_fmri2(''add_select'');');
        % list of selected component
        pos = [160 5 45 25];
        h1 = uicontrol(...
            'style','popupmenu',...
            'Parent',f_main, ...
            'Units','pixels', ...
            'BackgroundColor',RGB_mainwin, ...
            'FontSize',text_size, ...
            'ForegroundColor',RGB_text, ...
            'Position',pos, ...
            'string','0',...
            'Tag','select_list',...
            'callback','map_on_fmri2(''goto_select_list'');');
        % add items in select_list according to FMRI.icacomp;
        if ~isempty(FMRI.icacomp),
            set(h1,'string',num2str(sort(FMRI.icacomp)));
        end
        % unmark button for select component
        pos = [210 5 65 25];
        h1 = uicontrol(...
            'style','pushbutton',...
            'Parent',f_main, ...
            'Units','pixels', ...
            'BackgroundColor',RGB_button, ...
            'FontSize',text_size, ...
            'ForegroundColor',RGB_text, ...
            'Position',pos, ...
            'String','UnMark', ...
            'Tag','but_unmark_select',...
            'callback','map_on_fmri2(''remove_select'');');
         
        % name of button group 3 Select Componnet
        pos = [280 30 150 20];
        h1 = uicontrol(...
            'style','text',...
            'Parent',f_main, ...
            'Units','pixels', ...
            'BackgroundColor',RGB_mainwin, ...
            'FontSize',text_size, ...
            'ForegroundColor',RGB_text, ...
            'Position',pos, ...
            'horizontalalign','left',...
            'String','Reject Components');
        % mark button for rejecting component
        pos = [280 5 45 25];
        h1 = uicontrol(...
            'style','pushbutton',...
            'Parent',f_main, ...
            'Units','pixels', ...
            'BackgroundColor',RGB_button, ...
            'FontSize',text_size, ...
            'ForegroundColor',RGB_text, ...
            'Position',pos, ...
            'String','Mark', ...
            'Tag','but_mark_reject',...
            'callback','map_on_fmri2(''add_reject'');');
        % list of rejected component
        pos = [330 5 45 25];
        h1 = uicontrol(...
            'style','popupmenu',...
            'Parent',f_main, ...
            'Units','pixels', ...
            'BackgroundColor',RGB_mainwin, ...
            'FontSize',text_size, ...
            'ForegroundColor',RGB_text, ...
            'Position',pos, ...
            'string','0',...
            'Tag','reject_list',...
            'callback','map_on_fmri2(''goto_reject'');');
        % unmark button for reject component
        pos = [380 5 65 25];
        h1 = uicontrol(...
            'style','pushbutton',...
            'Parent',f_main, ...
            'Units','pixels', ...
            'BackgroundColor',RGB_button, ...
            'FontSize',text_size, ...
            'ForegroundColor',RGB_text, ...
            'Position',pos, ...
            'String','UnMark', ...
            'Tag','but_unmark_reject',...
            'callback','map_on_fmri2(''remove_reject'');');
         
        % name of button group 4 Select Componnet
        pos = [450 30 100 20];
        h1 = uicontrol(...
            'style','text',...
            'Parent',f_main, ...
            'Units','pixels', ...
            'BackgroundColor',RGB_mainwin, ...
            'FontSize',text_size, ...
            'ForegroundColor',RGB_text, ...
            'Position',pos, ...
            'horizontalalign','left',...
            'String','Time Course');
        % list of time course displays
        str = {'None','Comp.','ROA','Comp+ROA','BOLD Image'};
        pos = [450 5 110 25];
        h1 = uicontrol(...
            'style','popupmenu',...
            'Parent',f_main, ...
            'Units','pixels', ...
            'BackgroundColor',RGB_mainwin, ...
            'FontSize',text_size, ...
            'ForegroundColor',RGB_text, ...
            'Position',pos, ...
            'string',str,...
            'Tag','list_tc',...
            'callback','map_on_fmri2(''tc_clicked'');');
        
        % operations popup menu
        pos = [565 30 100 20];
        h1 = uicontrol(...
            'style','text',...
            'Parent',f_main, ...
            'Units','pixels', ...
            'BackgroundColor',RGB_mainwin, ...
            'FontSize',text_size, ...
            'ForegroundColor',RGB_text, ...
            'Position',pos, ...
            'horizontalalign','left',...
            'String','Operations');
        % list of time course displays
        str = {'--','PVAF vs Zth',...
	      'ER-BOLD'};
        pos = [565 5 110 25];
        h1 = uicontrol(...
            'style','popupmenu',...
            'Parent',f_main, ...
            'Units','pixels', ...
            'BackgroundColor',RGB_mainwin, ...
            'FontSize',text_size, ...
            'ForegroundColor',RGB_text, ...
            'Position',pos, ...
            'string',str,...
            'Tag','list_operation',...
            'callback','map_on_fmri2(''op_clicked'');');
         
        % create control for colorbar
        pos = [710 240 20 212];
        h1 = axes(...
            'Parent',f_main, ...
            'units','pixels',...
            'position',pos,...
            'Color',RGB_mainwin,...
            'box','on',...
            'Tag','image_ax',...
            'xtick',[],...
            'xticklabel',[],...
            'ytick',[],...
            'yticklabel',[]);
        % plot colorbar
        cb = linspace(65,128,64);
        cb = cb';
        b = repmat(cb,1,5);
        kk = image(flipud(b));
        set(h1,'xtick',[]);
        set(h1,'ytick',linspace(1,64,10));
        cb = linspace(8.0,1.5,5);
        cb = [cb linspace(-1.5,-8.0,5)];
        cb = num2str(cb','%4.1f');
        i = [2:4 7:9];
        for j=1:size(cb,2),
            cb(i,j) = ' ';
        end
        set(h1,'yticklabel',cb);
        fmrlab_color(2);
        cb = h1;
         
        % create edit for z_limit
        pos = [685 210 50 25];
        h1 = uicontrol(...
            'style','edit',...
            'string','8.0',...
            'position',pos,...
            'backgroundcolor',RGB_mainwin,...
            'foregroundcolor',RGB_text,...
            'tag','ed_lim',...
            'fontsize',text_size,...
            'callback','map_on_fmri2(''refresh'');');
        % create edit for z-threshold
        pos = [685 180 50 25];
        h1 = uicontrol(...
            'style','edit',...
            'string','1.5',...
            'position',pos,...
            'backgroundcolor',RGB_mainwin,...
            'foregroundcolor',RGB_text,...
            'tag','ed_thre',...
            'fontsize',text_size,...
            'callback','map_on_fmri2(''refresh'');');
        % create pgup and pgdn buttons
        pos = [685 55 50 25];
        h1 = uicontrol(...
            'style','pushbutton',...
            'string','PgDn',...
            'position', pos,...
            'backgroundcolor',RGB_button,...
            'foregroundcolor',RGB_text,...
            'callback','map_on_fmri2(''pgdn'');');
        if FMRI.dime_z < 15,
            set(h1,'enable','off');
        end
        pos = [685 85 50 25];
        h1 = uicontrol(...
            'style','pushbutton',...
            'string','PgUp',...
            'position', pos,...
            'backgroundcolor',RGB_button,...
            'foregroundcolor',RGB_text,...
            'callback','map_on_fmri2(''pgup'');');
        if FMRI.dime_z < 15,
            set(h1,'enable','off');
        end
        pos = [685 110 50 20];
        h1 = uicontrol(...
            'style','text',...
            'string',['of ' num2str(FMRI.dime_z)],...
            'position', pos,...
            'backgroundcolor',RGB_mainwin,...
            'foregroundcolor',RGB_text,...
            'tag','total_slice');
        if FMRI.dime_z < 15,
            set(h1,'visible','off');
        end
        pos = [685 130 50 20];
        h1 = uicontrol(...
            'style','text',...
            'string','1 : 15',...
            'position', pos,...
            'backgroundcolor',RGB_mainwin,...
            'foregroundcolor',RGB_text,...
            'tag','slice_range');
        if FMRI.dime_z < 15,
            set(h1,'visible','off');
        end
        
        % make 15 image displays
        ax = [];
        axw = 135;
        for i=1:3,
            for j=1:5,
                pos(1) = (j-1)*axw+5;
                pos(2) = 462 - i*axw;
                pos(3) = axw;
                pos(4) = axw;
                h1 = axes(...
                    'Parent',f_main, ...
                    'units','pixels',...
                    'position',pos,...
                    'Color','k', ...
                    'box','on',...
                    'Tag','image_ax',...
                    'xtick',[],...
                    'xticklabel',[],...
                    'ytick',[],...
                    'yticklabel',[]);
                fmrlab_color(2);
                ax = [ax h1];
            end
        end
         
        % save necessary information in userdata
        userdata.ax = ax;
        userdata.cb = cb;
        userdata.start_slice = 1;
        set(f_main,'userdata',userdata);
         
        % show images of component #1 to begin with
        display_component_map;
    else
        figure(f_main);
        map_on_fmri2('fresh');
    end
     
case 'tc_clicked'
	a = findobj('tag','list_tc');
	b = get(a,'value');
	if b == 1,
		f = findobj('name','Component Time Courses');
		if ~isempty(f),
			close(f);
		end
		f = findobj('name','BOLD Image Plot');
		if ~isempty(f),
			close(f);
		end
	else
		display_component_map;
	end
    
case 'close',
    a = findobj('tag','select_list');
    b = str2num(get(a,'string'));
    if b(1) ~= 0,
        FMRI.icacomp = b;
    else
        FMRI.icacomp = [];
    end
    a = findobj('tag','reject_list');
    b = str2num(get(a,'string'));
    if b(1) ~= 0,
        FMRI.rejcomp = b;
    else
        FMRI.rejcomp = [];
    end
    f_main = findobj('name',['Display Component ROA on Functional' ...
		    ' Variance Images']);
    f_time = findobj('name','Component Time Courses');
    if ~isempty(f_time),
      delete(f_time);
    end
    f_bold = findobj('name','BOLD Image Plot');
    if ~isempty(f_bold)
      delete(f_bold);
    end

    if ~isempty(f_main),
      delete(f_main);
    end
    
case 'backward'
    a = findobj('tag','ed_component');
    b = str2num(get(a,'string'));
    if b==1,
        b = size(FMRI.icaweight,1);
    else
        b = b-1;
    end
    set(a,'string',num2str(b));
    display_component_map;
    
case 'foreward'
    a = findobj('tag','ed_component');
    b = str2num(get(a,'string'));
    if b == size(FMRI.icaweight,1),
        b = 1;
    else
        b = b+1;
    end
    set(a,'string',num2str(b));
    display_component_map;
    
case 'refresh',
    display_component_map;
    
case 'goto_select_list',
    a = findobj('tag','select_list');
    b = get(a,'string');
    c = get(a,'value');
    c = str2num(b(c,:));
    a = findobj('tag','ed_component');
    set(a,'string',num2str(c));
    display_component_map;
    
case 'add_select',
    a = findobj('tag','ed_component');
    b = str2num(get(a,'string'));
    a = findobj('tag','select_list');
    c = get(a,'string');
    if str2num(c(1,:)) == 0,
        %disp('no items in the list');
        set(a,'string',num2str(b));
	FMRI.icacomp = b;
    else
        d = str2num(c);
        d = [d;b];
	FMRI.icacomp = sort(d(:));
        set(a,'string',num2str(sort(d)));
    end
    
case 'remove_select',
    % if no items in the list, warning
    a = findobj('tag','select_list');
    b = get(a,'string');
    if str2num(b(1,:)) == 0,
        fmrlab_messagedlg('map_on_fmri2()',...
            'Error:',...
            'No items in the list!');
        return;
    end
    % if the select item doesn't match the current displayed 
    % component, warning
    c = str2num(b);
    d = get(a,'value');
    aa = findobj('tag','ed_component');
    bb = str2num(get(aa,'string'));
    if bb ~= c(d);
        fmrlab_messagedlg('map_on_fmri2()',...
            'Error:',...
            'Please select the component you want to delete from the Select List first');
        return;
    end
    c(d) = [];
    FMRI.icacomp = c(:);
    if isempty(c),
        set(a,'string','0');
    else
        set(a,'string',num2str(c));
        set(a,'value',1);
    end

case 'goto_reject',
    a = findobj('tag','reject_list');
    b = get(a,'string');
    c = get(a,'value');
    c = str2num(b(c,:));
    a = findobj('tag','ed_component');
    set(a,'string',num2str(c));
    display_component_map;
    
case 'add_reject',
    a = findobj('tag','ed_component');
    b = str2num(get(a,'string'));
    a = findobj('tag','reject_list');
    c = get(a,'string');
    if str2num(c(1,:)) == 0,
        %disp('no items in the list');
        set(a,'string',num2str(b));
	FMRI.rejcomp = b;
    else
        d = str2num(c);
        d = [d;b];
	FMRI.rejcomp = sort(d(:));
        set(a,'string',num2str(sort(d)));
    end
    
case 'remove_reject',
    % if no items in the list, warning
    a = findobj('tag','reject_list');
    b = get(a,'string');
    if str2num(b(1,:)) == 0,
        fmrlab_messagedlg('map_on_fmri2()',...
            'Error:',...
            'No items in the list!');
        return;
    end
    % if the select item doesn't match the current displayed 
    % component, warning
    c = str2num(b);
    d = get(a,'value');
    aa = findobj('tag','ed_component');
    bb = str2num(get(aa,'string'));
    if bb ~= c(d);
        fmrlab_messagedlg('map_on_fmri2()',...
            'Error:',...
            'Please select the component you want to delete from the Reject List first');
        return;
    end
    c(d) = [];
    FMRI.rejcomp = c(:);
    if isempty(c),
        set(a,'string','0');
    else
        set(a,'string',num2str(c));
        set(a,'value',1);
    end
    
case 'op_clicked',
    a = findobj('tag','list_operation');
    b = get(a,'value');
    switch(b)
     case 2
      pvaf_vs_thre;
     case 3
      er_bold;
    end
    
case 'pgdn',
    a = findobj('name','Display Component ROA on Functional Variance Images');
    userdata = get(a,'userdata');
    if userdata.start_slice == (FMRI.dime_z-14),
        fmrlab_messagedlg('map_on_fmri2()',...
            'Warning:',...
            'No more slices to show.');
        return;
    else
        userdata.start_slice = userdata.start_slice+1;
        set(a,'userdata',userdata);
        a = findobj('tag','slice_range');
        set(a,'string',[num2str(userdata.start_slice) ' : ' num2str(userdata.start_slice+14)]);
        display_component_map;
    end
    
case 'pgup',
    a = findobj('name','Display Component ROA on Functional Variance Images');
    userdata = get(a,'userdata');
    if userdata.start_slice == 1,
        fmrlab_messagedlg('map_on_fmri2()',...
            'Warning:',...
            'First slice already');
        return;
    else
        userdata.start_slice = userdata.start_slice-1;
        set(a,'userdata',userdata);
        a = findobj('tag','slice_range');
        set(a,'string',[num2str(userdata.start_slice) ' : ' num2str(userdata.start_slice+14)]);
        display_component_map;
    end
end
  
  
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

  
function display_component_map()
global FMRI
RGB_mainwin = [1 .984 .895];
RGB_background = [0.860 0.824 0.618];
RGB_button = [0.860 0.824 0.618];
RGB_text = [0.333 0.283 0.003];
if isunix,
    text_size = 14;
else
    text_size = 12;
end
f_main = findobj('name',['Display Component ROA on Functional' ...
    ' Variance Images']);
if f_main == 0,
    disp('Cannot find main window');
    return;
end
userdata = get(f_main,'userdata');
% form base image
avg_image = sqrt(std(FMRI.data));
base_image = zeros(FMRI.dime_x*FMRI.dime_y*FMRI.dime_z,1);
for i=1:length(FMRI.idx),
    base_image(FMRI.idx(i)) = avg_image(i);
end
base_image = fmrlab_scale(base_image,1,64);

% find the displaying parameters
a = findobj(f_main,'tag','ed_thre');
uthre = str2num(get(a,'string'));
lthre = -1 * uthre;
a = findobj(f_main,'tag','ed_lim');
ulim = str2num(get(a,'string'));
llim = -1 * ulim;
% update ticks of colorbar
cb = linspace(ulim,uthre,5);
cb = [cb linspace(lthre,llim,5)];
cb = num2str(cb','%4.1f');
i = [2:4 7:9];
for j=1:size(cb,2),
    cb(i,j) = ' ';
end
set(userdata.cb,'yticklabel',cb);
  
% get current component
a = findobj(f_main,'tag','ed_component');
comp = str2num(get(a,'string'));
% calculating component map
W = FMRI.icaweight * FMRI.icasphere;
invW = pinv(W);
u = W * FMRI.data;
temp_data = u(comp, :);
mean_temp_data = mean(temp_data);
std_temp_data = std(temp_data);
temp_data = (temp_data-mean_temp_data)/std_temp_data; %convert to z score
uidx = find(temp_data >= ulim);
temp_data(uidx) = ulim;
uidx = find(temp_data >= uthre);
umap = zeros(length(temp_data),1);
umap(uidx) = temp_data(uidx);
umap(uidx) = fmrlab_scale(umap(uidx),101,128,uthre,ulim);
map = zeros(FMRI.dime_x*FMRI.dime_y*FMRI.dime_z,1);
for i=1:length(FMRI.idx),
    map(FMRI.idx(i)) = umap(i);
end
img = max(base_image,map);

lidx = find(temp_data <= llim);
temp_data(lidx) = llim;
lidx = find(temp_data <= lthre);
lmap = zeros(length(temp_data),1);
lmap(lidx) = temp_data(lidx);
lmap(lidx) = fmrlab_scale(lmap(lidx),65,92,llim,lthre);
map = zeros(FMRI.dime_x*FMRI.dime_y*FMRI.dime_z,1);
for i=1:length(FMRI.idx),
    map(FMRI.idx(i)) = lmap(i);
end
img = max(img,map);
img = reshape(img,FMRI.dime_x,FMRI.dime_y,FMRI.dime_z);

% display component maps
ii = min((userdata.start_slice-1)+15,FMRI.dime_z);
pp = 1;
for i=userdata.start_slice:ii
    axes(userdata.ax(pp));
    image(real(img(:,:,i))');
    axis off
    axis image
    pp = pp +1;
end
fmrlab_color(2);
  
% update the title bar according to the current component
a = findobj(f_main,'tag','title_string');
set(a,'string',['Independent Component #' num2str(comp)]);

% display time couse and bold image plots if specified
a = findobj(f_main,'tag','list_tc');
b = get(a,'value');
  
% display time course if b ~= 1
if b~=1,
    % calculate back-projected component time course and
    % + ROA mean time course
    ddd = zeros(size(invW,1),size(u,2));
    ddd = invW(:,comp) * u(comp,:);
    ddd = ddd(:,uidx)';
    bpctc = mean(ddd);
    ddd = FMRI.data(:,uidx);
    ddd = mean(ddd');
    bpctc = 100 * (bpctc-mean(bpctc)) / (mean(ddd));
    roatc = 100 * (ddd-mean(ddd))/(mean(ddd));

    % clean up the memory space before displaying figures
    clear u, invW;
    
    f_time = findobj('name','Component Time Courses');
    if isempty(f_time),
        pos = [5 40 735 200];
        f_time = [];
        f_time = figure(...
            'position',pos,...
            'numbertitle','off',...
            'menubar','none',...
            'color',RGB_mainwin,...
            'name','Component Time Courses');
    else
        figure(f_time);
        clf;
    end
    
    switch(b)
    case 2,
        if isempty(FMRI.tr) | FMRI.tr == 0,
	        plot(bpctc);
            set(gca,'fontsize',text_size);
	        xlabel('Scan Number');
       else
	        x1 = linspace(FMRI.tr,FMRI.timepoint*FMRI.tr,FMRI.timepoint);
	       plot(x1,bpctc);
	        set(gca,'fontsize',text_size);
        xlabel('Time (s)');
    end
        pos = get(gca,'position');
        set(gca,'position',[pos(1) pos(2)+pos(4)*0.1 pos(3) pos(4)*0.9]);
        legend('Comp. proj. to +ROA');
        ylabel('\Delta BOLD Signal (%)');
        set(gca,'ylim',[min(bpctc) max(bpctc)]);
        set(gca,'ytick',[min(bpctc) 0 max(bpctc)]);
        
    case 3,
        if isempty(FMRI.tr) | FMRI.tr == 0,
	        plot(roatc,'r');
	        set(gca,'fontsize',text_size);
	        xlabel('Scan Number');
        else
	        x1 = linspace(FMRI.tr,FMRI.timepoint*FMRI.tr,FMRI.timepoint);
	        plot(x1,roatc,'r');
	        set(gca,'fontsize',text_size);
	        xlabel('Time (s)');
        end
        pos = get(gca,'position');
        set(gca,'position',[pos(1) pos(2)+pos(4)*0.1 pos(3) pos(4)*0.9]);
        legend('+ROA mean time course');
        ylabel('\Delta BOLD Signal (%)');
        set(gca,'ylim',[min(roatc) max(roatc)]);
        set(gca,'ytick',[min(roatc) 0 max(roatc)]);
            
    case 4,
        if isempty(FMRI.tr) | FMRI.tr == 0,
	        plot(bpctc);
	        hold on
	        plot(roatc,'r');
	        set(gca,'fontsize',text_size);
	        xlabel('Scan Number');
        else
	        x1 = linspace(FMRI.tr,FMRI.timepoint*FMRI.tr, ...
		        FMRI.timepoint);
	        plot(x1,bpctc);
	        hold on
	        plot(x1,roatc,'r');
	        set(gca,'fontsize',text_size);
	        xlabel('Time (s)');
        end
        pos = get(gca,'position');
        set(gca,'position',[pos(1) pos(2)+pos(4)*0.1 pos(3) pos(4)*0.9]);
        legend('Comp. proj. to +ROA','+ROA mean time course');
        ylabel('\Delta BOLD Signal (%)');
        set(gca,'ylim',[min(min(roatc),min(bpctc))...
		    max(max(roatc),max(bpctc))]);
        set(gca,'ytick',[min(min(roatc),min(bpctc))...
		    0 max(max(roatc),max(bpctc))]);
        v = 100*((1-var(roatc-bpctc)/var(roatc)));
        %ti = text(0.75*max(x1),...
        %    1.2*min(str2num(get(gca,'yticklabel'))),...
        %    [sprintf('p.v.a.f. = %5.2f',v) '%']);
        ti = textsc(0.85,0.05,[sprintf('p.v.a.f. = %5.2f',v) '%']);
        set(ti,'fontsize',text_size,'color',[1 0 1]);

    
    case 5,
        if isempty(FMRI.tr) | FMRI.tr == 0,
	        plot(bpctc);
	        set(gca,'fontsize',text_size);
	        xlabel('Scan Number');
        else
	        x1 = linspace(FMRI.tr,FMRI.timepoint*FMRI.tr,FMRI.timepoint);
	        plot(x1,bpctc);
	        set(gca,'fontsize',text_size);
	        xlabel('Time (s)');
        end
        pos = get(gca,'position');
        set(gca,'position',[pos(1) pos(2)+pos(4)*0.1 pos(3) pos(4)*0.9]);
        legend('Comp. proj. to +ROA');
        ylabel('\Delta BOLD Signal (%)');
        set(gca,'ylim',[min(bpctc) max(bpctc)]);
        set(gca,'ytick',[min(bpctc) 0 max(bpctc)]);
       
        % find bold image plot window
        f_bold = findobj('name','BOLD Image Plot');
        if isempty(f_bold),
            pos = [750 40 270 350];
            f_bold = [];
            f_bold = figure(...
                'position',pos,...
                'numbertitle','off',...
                'menubar','none',...
                'color',RGB_mainwin,...
                'name','BOLD Image Plot');
        else
            figure(f_bold);
            clf;
        end
       
        % enter FMRI.soa if missing
        if length(FMRI.soa) == 0,
	        prompt = {'Trial onsets (scan numbers):',...
		        'Trial scans (in scans):'};
	        defs = {num2str(FMRI.soa), num2str(FMRI.trial_length)};
	        BOLD_err = 0;
	        a = fmrlab_inputdlg(prompt,'BOLD-image plot parameters',1,defs);
	        if ~isempty(a),
	            FMRI.soa = str2num(char(a(1)));
	            if length(FMRI.soa)==1 | length(FMRI.soa)==0,
	                fmrlab_messagedlg('Component ROAs on fMRI',...
			            'Error:',...
			            'Too few trials');
	                BOLD_err = 1;
	            end
   
	            for i=1:length(FMRI.soa),
	            if FMRI.soa(i) > FMRI.timepoint | FMRI.soa(i) < 1,
	                fmrlab_messagedlg('Component ROAs on fMRI',...
				        'Error:',...
				        'Onset scan number exceeds total scan number');
	                BOLD_err = 1;
	                break;
	            end
	        end
	   
	        FMRI.trial_length = str2num(char(a(2)));
	        if FMRI.trial_length == 0,
	            fmrlab_messagedlg('Component ROAs on fMRI',...
			        'Error:',...
			        'Trial length cannot be zero');
	            BOLD_err = 1;
	        end
	   
	        for i=1:length(FMRI.soa),
	            temp = FMRI.trial_length + FMRI.soa(i) - 1;
	            if temp > FMRI.timepoint,
	                fmrlab_messagedlg('Component ROAs on fMRI',...
				        'Error:',...
				        'Trial length exceeds the total scan number');
	                BOLD_err = 1;
	                break;
	            end
	        end
	    else      
	        BOLD_err = 1;
    end
       
	    if BOLD_err == 1,
	        a = findobj(f_main,'tag','list_tc');
	        set(a,'value',1);
	        close(f_time);
	        close(f_bold);
	        return;
        end
    end
	          
    a = []; 
    for i=1:length(FMRI.soa),
	    a = [a bpctc(FMRI.soa(i):FMRI.soa(i)+FMRI.trial_length-1)];
    end
    boldimage(a,ones(length(FMRI.soa),1),...
		 0:FMRI.tr:FMRI.tr*(FMRI.trial_length-1),...
		 ['Component #' num2str(comp)],1,1,'erp','nosort','noplot','cbar');
end
 
end
  