% mutual_information() - Estimate mutual information of two signals
%                        with independent pairs of samples using 
%                        'Mean Square Error Estimation' approach
%
% Usage:
%   >> mi = mutual_information(x,y);
%
% Inputs:
%   x - vector of first independent measurement
%   y - vector of second independent measurement
%
% output:
%   mi      - estimation of mutual information
%
% Author: Jeng-Ren Duann, CNL/Salk Inst. & SCCN/INC/UCSD, 2002
%
% See also:
%
% Notes: - the lengths of vector x and y should be identical
%        - bin number used in this estimation is cube root of the
%          length of the input array
   
% Copyright (C) 2002 Jeng-Ren Duann, Salk Institute, duann@salk.edu
%
% This program is free software; you can redistribute it and/or
% modify it under the terms of the GNU General Public License as
% published by the Free Software Foundation; either version 2 of
% the License, or (at your option) any later version.
%
% This program is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU General Public License for more details.
%
% You should have received a copy of the GNU General Public License
% along with this program; if not, write to the Free Software
% Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
%
% Author: Jeng-Ren Duann, CNL/Salk and INC/UCSD, 2002-12-10 

% $Log: mutual_information.m,v $
% Revision 1.1  2003/02/06 19:12:42  duann
% Initial revision
%

function mi = mutual_information(x,y)
  
% calculating joint histogram of x and y
  % 1. Check if the length of x and y are equal
  sizex = size(x);
  sizey = size(y);
  if sizex(1) ~= sizey(1) | sizex(2) ~= sizey(2),
    disp('Lengths of x and y are not identical...!');
    return
  end
  % 2. determining the bins according to the length of data
  bins = ceil(length(x)^(1/3));
  % 3. determing the dynamic range to rescale the input data
  maxx = max(x);
  minx = min(x);
  maxy = max(y);
  miny = min(y);
  stepx = (maxx-minx)/(length(x)-1);
  stepy = (maxy-miny)/(length(y)-1);
  lx = minx-stepx/2;
  ux = maxx+stepx/2;
  ly = miny-stepy/2;
  uy = maxy+stepy/2;
  % 4. according to the dynamic range of each data, categorize
  %    each data sample to their bins
  xx = round((x-lx)/(ux-lx)*bins+1/2);
  yy = round((y-ly)/(uy-ly)*bins+1/2);
  % 5. calculating the joint histogram
  joint_hist = zeros(bins,bins);
  for i=1:length(xx),
    joint_hist(xx(i),yy(i)) = joint_hist(xx(i),yy(i))+1;
  end
  
% according to the joint histogram calculate the mutual 
% information
  mi = 0;
  std_err = 0;
  cnt = 0;
 
  hy = sum(joint_hist);
  hy = hy / sum(hy);
  hx = sum(joint_hist');
  hx = hx / sum(hx);
  joint_hist = joint_hist/sum(joint_hist(:));
  hxx = 0;
  for i=1:length(hx),
    if hx(i) ~= 0,
      logf = log(hx(i));
    else
      logf = 0;
    end
    hxx = hxx + hx(i) * logf;
  end
  hxx = -hxx;
  hyy = 0;
  for i=1:length(hy),
    if hy(i) ~= 0,
      logf = log(hy(i));
    else
      logf = 0;
    end
    hyy = hyy + hy(i) * logf;
  end
  hyy = -hyy;
  hxy = 0;
  for i=1:size(joint_hist,1),
    for j=1:size(joint_hist,2)
      if joint_hist(i,j) ~= 0,
	logf = log(joint_hist(i,j));
      else
	logf = 0;
      end
      hxy = hxy + joint_hist(i,j) * logf;
    end
  end
  hxy = -hxy;
  mi = hxx + hyy - hxy;
  mi = 2*mi / (hxx + hyy);
  
 
