% riamap_ui() - 
%    1. load the specified result file (e.g. 'result1.mat') and calculate
%       component maps of all independent components
%    2. convert the component maps to Z score
%    3. for each voxel, select the maximal Z score and the component contribute
%       to that Z score (Z score map and component number map)
%    4. in component number map, select the region of interest (ROI), it will
%       popup a window showing the mean time course by averaging raw fMRI time
%       series within the ROI. in the same time, it will give you the percentage
%       variance accounted maximally for the mean time course by the specified
%       component within the ROI.
%    5. the ROI you specify will be display on the upper left corner.
%    6. use left button to select the region of interest, use right button to 
%       quit this program
%
% Usage:
%    >> roamap_ui(train_data,train_idx,wgt,sph,dime,z_thre,tm,comp_opt);
%
% Inputs:
%    train_data  - 4D fMRI data within the brain area used by ICA training
%    train_idx   - compressed coordinate (1D) of the voxels within the brain area
%    wgt         - weight matrix obtained by ICA training
%    sph         - sphere matrix obtained by ICA training
%    dime        - matrix contains the information for image dimension, should be 
%                  [image_width image_height image_depth]
%    z_thre      - threshold used to construct the Z score map
%    tm          - tick label for time interval instead of time
%                  point
%    comp_opt    - component option 
%                  1: all the components; 
%                  2: excluding the rejected components; (FMRI.rejcomp) 
%                  3: using only the selected components (FMRI.icacomp)
% 
% Outputs:
%
% Author: Jeng-Ren Duann, CNL/Salk Inst. & SCCN/INC/UCSD, 2002
%
% See also: roaproj_ui(), roatc_ui()
 
% --
% Copyright (C) 2002 Jeng-Ren Duann, Salk Institute, duann@salk.edu
%
% This program is free software; you can redistribute it and/or
% modify it under the terms of the GNU General Public License as
% published by the Free Software Foundation; either version 2 of
% the License, or (at your option) any later version.
%
% This program is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU General Public License for more details.
%
% You should have received a copy of the GNU General Public License
% along with this program; if not, write to the Free Software
% Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

% $Log: roamap_ui.m,v $
% Revision 1.1  2003/02/06 19:15:02  duann
% Initial revision
%

function roamap_ui(train_data,train_idx,wgt,sph,dime,z_thre,tm,comp_opt)

% Jeng-Ren Duann, CNL/Salk Inst. 2001-10-08
%    add input argument 'comp_opt', J.-R. 2002-0520

global FMRI

if length(FMRI) == 0,
  fmrlab_messagedlg('Component ROA (z value)',...
	     'Error:',...
	     ['No current data set exists. Please load a ''.fmr''' ...
	      ' data files first']);
  return;
end

RGB_mainwin = [1 .984 .895];
RGB_background = [0.860 0.824 0.618];
RGB_button = [0.860 0.824 0.618];
RGB_text = [0.333 0.283 0.003];

if isunix,
    text_size = 14;
else
    text_size = 12;
end

if nargin < 1,
  train_data = FMRI.data;
end
if nargin < 2,
  train_idx = FMRI.idx;
end
if nargin < 3,
  wgt = FMRI.icaweight;
end
if nargin < 4,
  sph = FMRI.icasphere;
end
if nargin < 5,
  dime = [FMRI.dime_x FMRI.dime_y FMRI.dime_z];
end
if nargin < 6,
  prompt = 'Set threshold (1.5):';
  a = fmrlab_inputdlg(prompt,'ROAMAP Display',1,{'1.5'});
  if isempty(a),
    %errordlg('Stopped by user!');
    return;
  end
  z_thre = str2num(char(a));
end
if nargin < 7,
  tm = linspace(0,FMRI.tr*(FMRI.timepoint-1),FMRI.timepoint);
end
if nargin < 8,
  prompt = sprintf(['Components to Include:\n  1: all\n  2: exclude' ...
		    [' rejected components\n  3: selected components' ...
		     ' only']]);
  a = fmrlab_inputdlg(prompt,'ROAMAP Display',1,{'1'});
  if isempty(a),
    %errordlg('Stopped by user!');
    return;
  end
  comp_opt = str2num(char(a));
end

% plot z score map for all ica components
W = wgt*sph;
progressbar(0.2); drawnow
u = W*train_data;
progressbar(0.4); drawnow
invW = pinv(W);
progressbar(0.8); drawnow

sz = size(u);
switch comp_opt,
 case 1
  disp('Display all components');
  comp_to_show = 1:sz(1);
 case 2
  disp('Exclude rejected components');
  comp_to_show = 1:sz(1);
  comp_to_show(FMRI.rejcomp) = [];
  sz(1) = sz(1)-length(FMRI.rejcomp);
 case 3
  disp('Display selected components only');
  comp_to_show = FMRI.icacomp;
  sz(1) = length(comp_to_show);
end

map = zeros(sz(1),sz(2));
for i=1:length(comp_to_show)
  mu = mean(u(comp_to_show(i),:));
  su = std(u(comp_to_show(i),:));
  map(i,:) = (u(comp_to_show(i),:)-mu)/su;
end  
im = zeros(1,sz(2));
im2 = zeros(1,sz(2));
for i=1:sz(2)
  [im(i),im2(i)] = max(abs(map(:,i)));
end
mm1 = zeros(dime(1)*dime(2)*dime(3),1);
mm2 = 101*ones(dime(1)*dime(2)*dime(3),1);
for i=1:length(train_idx)
  mm1(train_idx(i)) = im(i);
  mm2(train_idx(i)) = comp_to_show(im2(i));
end
ii = find(mm1 < z_thre);
mm2(ii) = 101;
mm1 = reshape(mm1,dime(1),dime(2),dime(3));
mm2 = reshape(mm2,dime(1),dime(2),dime(3));
progressbar(1); drawnow
pause(0.5)
progressbar(nan);

f_zmap = figure(...
    'name','Tiling Brain by ICA Components: Maximal Z Map',...
    'numbertitle','off',...
    'color',RGB_mainwin,...
    'toolbar','none',...
    'menubar','none');
disp_col = ceil(sqrt(dime(3)));
disp_row = ceil(dime(3)/disp_col);
for i=1:dime(3)
  subplot(disp_row,disp_col,i);
  imagesc((mm1(:,:,i)'));
  axis image; axis off
end
cbar;
ti = textsc(0.5,0.95,'Largest Z Value Map');
set(ti,'fontsize',text_size);

f_comp = figure(...
    'name','Independent Component Regions of Activity',...
    'numbertitle','off',...
    'color',RGB_mainwin,...
    'toolbar','none',...
    'menubar','none');
for i=1:dime(3)
  subplot(disp_row,disp_col,i);
  imagesc(mm2(:,:,i)');
  axis image; axis off
end
colorm = colormap;
colormap(flipud(colorm));
cb = cbar;
pos = get(cb,'position');
set(cb,'position',[pos(1)+pos(3) pos(2) pos(3) pos(4)]);
ti = textsc(pos(1),pos(2)+pos(4)/2,'Component Number');
set(ti,'rotation',90);
ti = textsc(0.5,0.95,'Component with Largest Weight at Each Voxel');
set(ti,'fontsize',text_size);
ti = textsc(0.5,0.08,'Press left button to blow up and specify ROI');
ti = textsc(0.5,0.04,'Press right button to quit');

figure(f_comp);
while 1
  [my,mx,mb] = ginput(1);
  if mb == 3,
    break;
  end
  if mb == 1,
    a = get(gco,'cdata');
    tempf = figure(...
	'name','Select Component of Interest',...
	'numbertitle','off',...
	'color',RGB_mainwin,...
	'toolbar','none',...
	'menubar','none');
    imagesc(a);
    axis image;
    axis off;
    cmap = colormap;
    colormap(flipud(cmap));
    ti = title('Click left button to specify component of interest');
    cbar;
%    [my,mx,mb] = ginput(1);
%    comp = a(round(mx),round(my));
    [mx,my,comp,b]=mousemoving(tempf);
    close(tempf);
    if b == 3,
      break;
    end
    if b == 1,
      if (comp<1 | comp>100)
        disp(['Area with no value is specified, try again...']);
	continue;
      else
        disp(['Component: ' int2str(comp)]);
        tm = linspace(0,FMRI.tr*(FMRI.timepoint-1),FMRI.timepoint);
        v = roatc_ui(train_data,train_idx,wgt,sph,dime,comp,z_thre,3,tm);
        set(gcf,'name','ROA-Mean Time Course: Largest-Contributing Components');
	b = zeros(size(a));
	ii = find(a==comp);
	b(ii) = 80;
        ii = find(a<101 & a~=comp);
        b(ii) = 20;
        sbplot(3,3,1);
        imagesc(b); axis image; axis off
        caxis([0 101]);
	%cmap = colormap;
	%colormap(flipud(cmap));

        % doing the roaproj stuff
        figure(...
	    'name','Component ROA-Mean Time Course: Largest Component Contribution',...
	    'numbertitle','off',...
	    'color',RGB_mainwin,...
	    'toolbar','none',...
	    'menubar','none');
        sbplot(4,4,[6 11]);
        imagesc(b); axis off; axis image
        ax = get(gca,'position');
	set(gca,'position',[ax(1) ax(2)+.12 ax(3) ax(4)]);
        caxis([0 101]);
        t1 = title(['IC  ' int2str(comp)]);
        set(t1,'fontsize',text_size+2);
	ax = axes('position',[ax(1)+0.1*ax(3) ax(2)-.22 0.8*ax(3) .22]);

        mm = u(comp,:);
        mm = (mm-mean(mm))/std(mm);

        ii = find(mm>=z_thre);
        c = (mean(train_data(:,ii)'));
        mean_c = mean(c);
        c = 100*(c-mean_c)/mean_c;
        plot(tm,c);
        set(gca, 'yticklabel',[]);
        [p,q] = sort(v);
        pos = [1 3 4 6];
        j = 1;
        r = c;
        ss = zeros(1,length(c));
        for i=length(q):-1:length(q)-3,
          u2 = zeros(sz(1),sz(2));
          u2(q(i),:) = u(q(i),:);
          data2 = invW(:,q(i))*u2(q(i),:); 
          tc2 = data2(:,ii)';
          mean_tc2 = (mean(tc2));
          mean_tc2 = 100*(mean_tc2 - mean(mean_tc2)) / mean_c;
          sbplot(2,3,pos(j));
          plot(tm,c);
          hold on
          plot(tm,mean_tc2,'r','linewidth',2);
          if (j==2 | j==4),
            set(gca,'yticklabel',[]);
          end
          if (j==1 | j==3),
            tx = get(gca,'ylabel');
            set(tx,'string','\Delta BOLD Signal (%)');
            set(tx, 'fontsize',text_size);
          end
          if (j==3 | j==4),
            tx = get(gca,'xlabel');
            set(tx,'string','Time (s)');
            set(tx, 'fontsize',text_size);
          end
          ti = sprintf('IC%d      %4.1f%%',q(i),p(i));
          tx = title(ti);
          set(tx,'fontsize',text_size);
          r = r-mean_tc2;
          ss = ss + mean_tc2;
          j = j+1;

	  if comp == q(i),
	    set(get(gca,'title'),'fontweight','bold');
%            set(gca,'xcolor',[1 0 0]);
%            set(gca,'ycolor',[1 0 0]);
%            ca = get(gca,'ylabel');
%            set(ca,'color',[0 0 0]);
          end
        end
        axes(ax);
        hold on
        plot(tm,r,'k');
        plot(tm,ss,'r');
        ti = get(gca,'xlabel');
        set(ti,'string','Time (s)');
        set(ti, 'fontsize',text_size);

        vv = 100* (1-var(c-ss)/var(c));
        axes(ax);
        ax2 = axis;
        ti = sprintf('%4.1f%%',vv);
        tx = title(['All four combined  ' ti]);
        set(tx,'fontsize',text_size);
      end
    end

    figure(f_comp);
  end
end

