% roaproj_ui() - ROA projection with Component ROA-Mean Time
%                courses display with graphic
%                user interface (GUI) for user to input the
%                parameters.
%  
% Usage:
%    >>roaproj_ui(train_data,train_idx,wgt,sph,dime,compnum,slice_show,z_thre,tm);
%
% Inputs:
%    train_data - train data set, the voxels inside brain area
%    train_idx  - index indicates the coordinate of the voxels inside brain area
%    wgt        - weight matrix obtained from ICA training
%    sph        - sphere matric obtained from ICA training
%    dime       - matrix for image dimension ([image_width image_height depth])
%    compnum    - the component number used to select the ROA
%    slice_show - slice to be displayed in the figure
%    z_thre     - threshold used to find out the ROA
%    tm         - tick label for time interval instead of time point
% 
% Outputs:
%
% Author: Jeng-Ren Duann, CNL/Salk Inst. & SCCN/INC/UCSD, 2002
%
% See also: roamap_ui(), roatc_ui()

% --
% Copyright (C) 2002 Jeng-Ren Duann, Salk Institute, duann@salk.edu
%
% This program is free software; you can redistribute it and/or
% modify it under the terms of the GNU General Public License as
% published by the Free Software Foundation; either version 2 of
% the License, or (at your option) any later version.
%
% This program is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU General Public License for more details.
%
% You should have received a copy of the GNU General Public License
% along with this program; if not, write to the Free Software
% Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

% $Log: roaproj_ui.m,v $
% Revision 1.1  2003/02/06 19:15:11  duann
% Initial revision
%

function roaproj_ui(train_data,train_idx,wgt,sph,dime,compnum,slice_show,z_thre,tm)

% Jeng-Ren Duann, CNL/Salk Inst. 2001-10-04

global FMRI

if length(FMRI) == 0,
  fmrlab_messagedlg('Decompose a component ROA time course',...
		'Error:',...
                ['No current dataset exists. Please load ''.fmr'' data' ...
		 ' file first.']);
  return;
end

RGB_mainwin = [1 .984 .895];
RGB_background = [0.860 0.824 0.618];
RGB_button = [0.860 0.824 0.618];
RGB_text = [0.333 0.283 0.003];

if isunix,
    text_size = 14;
else
    text_size = 12;
end

if nargin < 1,
  train_data = FMRI.data;
end
if nargin < 2,
  train_idx = FMRI.idx;
end
if nargin < 3,
  wgt = FMRI.icaweight;
end
if nargin < 4,
  sph = FMRI.icasphere;
end
if nargin < 5,
  dime = [FMRI.dime_x FMRI.dime_y FMRI.dime_z];
end
if nargin < 6,
  prompt = ['Component number (1-N): [where N if number of components]'];
  prompt = {prompt};
  a = fmrlab_inputdlg(prompt,...
		  'Decompose a component ROA time course',1,{'1'});
  if isempty(a),
    %errordlg('Interrupted by user!');
    return;
  end
  compnum = str2num(char(a));
end
if nargin < 7,
  x = [1:dime(3)];
  defs = {int2str(x(ceil(length(x)/2)))};
  prompt = ['Select a slice to display (' int2str(x(ceil(length(x)/2))) '):'];
  prompt = {prompt};
  a = fmrlab_inputdlg(prompt,'Select Slice',1,defs);
  if isempty(a),
    %errordlg('Interrupted by user!');
    return;
  end
  slice_show = str2num(char(a));
end
if nargin < 8,
  prompt = ['Set threshold (1.5):'];
  a = fmrlab_inputdlg(prompt,'Set Threshold',1,{'1.5'});
  if isempty(a),
    %errordlg('Interrupted by user!');
    return;
  end
  z_thre = str2num(char(a));
end
if nargin < 9,
  tm = linspace(0,FMRI.tr*(FMRI.timepoint-1),FMRI.timepoint);
end

progressbar(0.1);
drawnow
W = wgt*sph;
progressbar(0.2);
drawnow
invW = pinv(W);
progressbar(0.3);
drawnow
u = W*train_data;
progressbar(0.5);
drawnow

sz = size(u);
map = zeros(sz(1),sz(2));
for i=1:sz(1)
  mu = mean(u(i,:));
  su = std(u(i,:));
  map(i,:) = (u(i,:)-mu)/su;
end

im = zeros(1,sz(2));
im2 = zeros(1,sz(2));
for i=1:sz(2)
  [im(i),im2(i)] = max(abs(map(:,i)));
end
mm1 = zeros(dime(1)*dime(2)*dime(3),1);
mm2 = zeros(dime(1)*dime(2)*dime(3),1);
for i=1:length(train_idx)
  mm1(train_idx(i)) = im(i);
  mm2(train_idx(i)) = im2(i);
end
ii = find(mm1 < z_thre);
mm2(ii) = 0;
mm1 = reshape(mm1,dime(1),dime(2),dime(3));
mm2 = reshape(mm2,dime(1),dime(2),dime(3));
a = mm2(:,:,slice_show);

progressbar(0.6);
drawnow

f_roaproj = figure(...
    'name','Component ROA-Mean Time Course: Largest Component Contribution',...
    'numbertitle','off',...
    'color',RGB_mainwin,...
    'toolbar','none',...
    'menubar','none');
ii = find(a>0 & a~=compnum);
if compnum == 1,
  a(ii) = 2;
  ii = find(a == compnum);
  a(ii) = 4;
else
  a(ii) = compnum/2;
end
sbplot(4,4,[6 11]);
imagesc(a'); axis off; axis image
ax = get(gca,'position');
set(gca,'position',[ax(1) ax(2)+.12 ax(3) ax(4)]);
if compnum == 1,
  caxis([0 8]);
else
  caxis([0 2*compnum]);
end
t1 = title(['IC  ' int2str(compnum)]);
set(t1,'fontsize',text_size+2);
ax = axes('position',[ax(1)+0.1*ax(3) ax(2)-.22 0.8*ax(3) .22]);
mm = u(compnum,:);
mm = (mm-mean(mm))/std(mm);
ii = find(mm>=z_thre);
c = mean(train_data(:,ii)');
mean_c = mean(c);
c = 100 * (c-mean_c) / mean_c;
if exist('tm')
  plot(tm,c);
else
  plot(c);
end
set(gca, 'yticklabel',[]);

progressbar(0.7);
drawnow

if exist('tm'),
  v = roatc_ui(train_data,train_idx,wgt,sph,dime,compnum,z_thre,3,tm);
else
  v = roatc_ui(train_data,train_idx,wgt,sph,dime,compnum,z_thre);
end

progressbar(0.95);
drawnow

figure(f_roaproj);
[p,q] = sort(v);
pos = [1 3 4 6];
j = 1;
r = c;
ss = zeros(1,length(c));

for i=length(q):-1:length(q)-3,
  data2 = invW(:,q(i))*u(q(i),:); 
  tc2 = data2(:,ii)';
  mean_tc2 = mean(tc2);
  mean_tc2 = 100 * (mean_tc2-mean(mean_tc2)) / mean_c;
  sbplot(2,3,pos(j));
  if exist('tm')
    plot(tm,c);
  else
    plot(c);
  end
  hold on
  if exist('tm'),
    plot(tm,mean_tc2,'r','linewidth',2);
  else
    plot(mean_tc2,'r','linewidth',2);
  end
  if (j==2 | j==4),
    set(gca,'yticklabel',[]);
  end
  if (j==1 | j==3),
    tx = get(gca,'ylabel');
    set(tx,'string','\Delta BOLD Signal (%)');
    set(tx, 'fontsize',text_size);
  end
  if (j==3 | j==4),
    tx = get(gca,'xlabel');
    if exist('tm'),
      set(tx,'string','Time (s)');
    else
      set(tx,'string','Scan Number');
    end
    set(tx, 'fontsize',text_size);
  end
  ti = sprintf('IC%d      %4.1f%%',q(i),p(i));
  tx = title(ti);
  set(tx,'fontsize',text_size);
  r = r-mean_tc2;
  ss = ss + mean_tc2;
  j = j+1;

  if q(i) == compnum,
    set(get(gca,'title'),'fontweight','bold');
%    set(gca,'xcolor',[1 0 0]);
%    set(gca,'ycolor',[1 0 0]);
%    ca = get(gca,'ylabel');
%    set(ca,'color',[0 0 0]);
  end
end
axes(ax);
hold on
if exist('tm')
  plot(tm,r,'k');
  plot(tm,ss,'r');
else
  plot(r,'k');
  plot(ss,'r');
end
ti = get(gca,'xlabel');
if exist('tm'),
  set(ti,'string','Time (s)');
else
  set(ti,'string','Scan Number');
end
set(ti, 'fontsize',text_size);

vv = 100* (1-var(c-ss)/var(c));
axes(ax);
ax2 = axis;
ti = sprintf('%4.1f%%',vv);
tx=title(['All four combined  ' ti]);
set(tx,'fontsize',text_size);

progressbar(1);
drawnow
pause(0.5);
progressbar(nan);
