% roatc_ui() - 
%    1. Per parameters of train_data, train_idx, wgt, and sph, calculating
%       the region of activity (ROA) according to the specified component
%       number. The ROA is selected according to the threshold of z score.
%    2. display the mean time course in the ROA.
%    3. back project all the ICA components onto the original space and over
%       plot on the mean time course of raw data within ROA
%    4. highlighten the component account for the variance maximally
%    5. return the percentage variance accounted for by all
%    components
%
% Usage:
%    >> [v,rmeantc] = roatc_ui(train_data,train_idx,wgt,sph,dime,compnum,...
%                     z_thre,comp_display,tm);
%
% Inputs:
%    train_data - voxels inside brain area which are used for ICA training
%    train_idx  - coordinates of the voxels inside the brain area
%    wgt        - weight matrix from ICA training
%    sph        - sphere matric from ICA training
%    dime       - matrix for image dimension ([image_width image_height depth])
%    compnum    - the component number used to select the ROA
%    z_thre     - threshold used to find out the ROA
%    comp_display - integer n, first n component which account for the percentage 
%                   variance maximally will be display in the background as black
%                   traces.
%    tm         - tick label for time interval instead of time point
%
% Outputs:
%    v       - one-dimensional array contained the information of percentage 
%              variance accounted for by all ICA components.
%    rmeantc - raw mean time course within the ROA
% 
% Author: Jeng-Ren Duann, CNL/Salk Inst. & SCCN/INC/UCSD, 2002
%
% See also: roamap_ui(), roaproj_ui()

% --
% Copyright (C) 2002 Jeng-Ren Duann, Salk Institute, duann@salk.edu
%
% This program is free software; you can redistribute it and/or
% modify it under the terms of the GNU General Public License as
% published by the Free Software Foundation; either version 2 of
% the License, or (at your option) any later version.
%
% This program is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU General Public License for more details.
%
% You should have received a copy of the GNU General Public License
% along with this program; if not, write to the Free Software
% Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

% $Log: roatc_ui.m,v $
% Revision 1.1  2003/02/06 19:15:19  duann
% Initial revision
%

function [v,rmeantc] = roatc_ui(train_data,train_idx,wgt,sph,dime,compnum,...
                       z_thre,comp_display,tm);

% Jeng-Ren Duann, CNL/Salk Inst. 2001-10-04

if nargin<8,
  comp_display = 3;
end

if nargin < 7,
  z_thre = 1.5;
end

RGB_mainwin = [1 .984 .895];

if isunix,
    text_size = 14;
else
    text_size = 12;
end

W = wgt*sph;
invW = pinv(W);
u = W*train_data;

map = u(compnum,:);
map = (map-mean(map))/std(map);
ii = find(map>=z_thre);
tc1 = mean(train_data(:,ii)');
grand_mean = mean(tc1);
tc1 = 100*(tc1 - mean(tc1))/grand_mean;
% tc1 is the time course from the raw data

h = figure(...
    'name','ROA-Mean Time Couese: Largest-Contributing Components',...
    'toolbar','none',...
    'numbertitle','off',...
    'menubar','none',...
    'color',RGB_mainwin);
if exist('tm')
  plot(tm,tc1,'k','linewidth',1);
else
  plot(tc1,'k','linewidth',1);
end
hold on

sz = size(u);
v = zeros(sz(1));
for i=1:sz(1)
  data2 = invW(:,i)*u(i,:); 
  tc2 = mean(data2(:,ii)');
  % tc2 is the time course from the back projection
  tc2 = 100 * (tc2-mean(tc2))/grand_mean;
  v(i) = 100*(1-(var(tc1-tc2)/var(tc1)));
end

[a,b] = sort(v);
j=1;
cols = ['b', 'g', 'm'];
fcols = [];
for i=sz(1):-1:sz(1)-comp_display+1
  u2 = zeros(sz(1),sz(2));
  u2(b(i),:) = u(b(i),:);
  data2 = invW(:,b(i))*u2(b(i),:); 
  tc2 = mean(data2(:,ii)');
  tc2 = 100 * (tc2-mean(tc2))/grand_mean;

  figure(h);
  hold on
  if b(i)==compnum,
    col = 'r';
    lw = 2;
    fcols = [fcols 'r'];
  else
    col = cols(j);
    lw = 0.5;
    fcols = [fcols cols(j)];
    j = j+1;
    if j>3,
      j = 3;
    end
  end
  if exist('tm'),
    plot(tm,tc2,col,'linewidth',lw);
  else
    plot(tc2,col,'linewidth',lw);
  end
end

u2 = zeros(sz(1),sz(2));
u2(compnum,:) = u(compnum,:);
data2 = invW(:,compnum)*u2(compnum,:); 
tc2 = mean(data2(:,ii)');
tc2 = 100 * (tc2-mean(tc2))/grand_mean;
figure(h)
hold on
if exist('tm'),
  plot(tm,tc2,'r','linewidth',2);
else
  plot(tc2,'r','linewidth',2);
end
ti = get(gca,'xlabel');
if exist('tm'),
  set(ti,'string','Time (s)');
else
  set(ti,'string','Scan Number');
end
set(ti,'fontsize',text_size);
ti = get(gca,'ylabel');
set(ti,'string','\Delta BOLD Signal (%)');
set(ti,'fontsize',text_size);

ax2 = axis;
ti = sprintf('IC%d ROA: %d Voxels\n    PVAF %5.2f%%',compnum,length(ii),v(compnum));
text(0.73*ax2(2)-ax2(1),ax2(3)+0.1*(ax2(4)-ax2(3)),ti);

ti = sprintf('Contribution of Largest Comp (%5.2f%%)',v(compnum));
ti = title(ti);
set(ti,'fontsize',text_size);

axes('position',[0 0 1 1])   
plot([0 1],[0 1],'k','linewidth',2);
for i=1:comp_display
  hold on
  plot([0 1],[0 1],fcols(i),'linewidth',2);
end

if b(sz(1))==compnum,  
  legend('ROA mean time course','Defining Component Projection*', ...
	 '2nd-Largest Component','3rd-Largest Component');
elseif b(sz(1)-1)==compnum,
  legend('ROA mean time course','Defining Component Projection',...
	 '2nd-Largest Component*','3rd-Largest Component');
else
  legend('ROA mean time course','Defining Component Projection',...
	 '2nd-Largest Component','3rd-Largest Component*');
end
  
axis([2 3 2 3])
axis off
