% boldimage() - image event_related fMRI optionally sorted on and/or aligned to 
%                 an input variable and smoothed by moving-average 
% Usage:
%   >> [outdata,outvar,outtrials,limits,axhndls,erp,amps,cohers,cohsig] = ...
%            boldimage(data,sortvar,times,'title',avewidth,decimate,option(s));
% Inputs:
%   data     - single-channel data matrix: (1,frames*trials) or (frames,trials)
%   sortvar  - vector variable to sort trials on (ntrials = length(sortvar))
%              Example sortvar = rts (in ms)
%   times    - vector of times in ms (frames=length(times)){def|0->[0:frames-1]}
%  'title'   - title string {default none}
%   avewidth - ntrials in moving average window (may be non-int) {def|0->1}
%   decimate - factor to decimate ntrials out by (may be non-int) {def|0->1}
%   option(s)- 'align',[time srate] -> time lock data to sortvar aligned to time
%                        (ms) (Hz) (time = 0 -> align to trimmed-median sortvar)
%            - 'nosort'-> don't sort data on sortvar {default: sort}
%            - 'noplot'-> don't plot sortvar {default: plot if in times range}
%            - 'limits',[lotime hitime minerp maxerp loamp hiamp locoher hicoher] 
%                        (can use nan for missing items and omit late items)
%            - 'caxis',[lo hi] -> set color axis limits {default: data bounds}
%            - 'cbar' -> plot color bar to right of erp-image {default no}
%            - 'erp' -> plot erp time average of the trials below the image
%            - 'coher',[freq srate] -> plot erp plus amp & coher at freq (Hz)
%                   or [freq srate alpha] -> add alpha coherence significance line
%                                           (alpha range: (0,0.1]) {default none}
% Outputs:
%   outdata  = (times,epochsout) data matrix (after smoothing)
%   outvar   = (1,epochsout)  sortvar vector (after smoothing)
%   outtrials= (1,epochsout)  smoothed trial numbers 
%   limits   = (1,8) array as in option 'limits' above (undefined -> nan)
%   axhndls  = vector of 1-4 plot axes handles
%   erp      = plotted ERP average
%   amp      = plotted amplitude time course
%   coher    = plotted coherence time course
%   cohsig   = coherence significance level
%
% Author: Jeng-Ren Duann, CNL/Salk Inst., 2002-09
%         adapted from erpimage() written by Tzyy-Ping Jung and
%         Scott Makeig
% See also: map_on_fmri(), map_on_struc()

% Copyright (C) Sigurd Enghoff, CNL / Salk Institute, La Jolla, 7/1998
%
% This program is free software; you can redistribute it and/or modify
% it under the terms of the GNU General Public License as published by
% the Free Software Foundation; either version 2 of the License, or
% (at your option) any later version.
%
% This program is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU General Public License for more details.
%
% You should have received a copy of the GNU General Public License
% along with this program; if not, write to the Free Software
% Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

% $Log: boldimage.m,v $
% Revision 1.1  2003/01/29 20:55:49  duann
% Initial revision
%

% Tzyy-Ping Jung & Scott Makeig, CNL / Salk Institute, La Jolla 3-2-98
%
% 3/5/98 added nosort option -sm
% 3/22/98 added colorbar ylabel, sym. range finding -sm
% 5/08/98 added noplot option -sm
% 6/09/98 added align, erp, coher options -sm
% 6/10/98 added limits options -sm
% 6/26/98 made number of variables output 8, as above -sm 
% 9/16/98 plot out-of-bounds sortvars at nearest times boundary -sm
% 10/27/98 added cohsig, alpha -sm
% 10/28/98 adjust maxamp, maxcoh computation -sm
% 11/08/00 adapted to display BOLD signal for fMRI analysis and
%          renamed as 'boldimage()' - JR

% Bugs:
% 'limits', [lotime hitime] does not work (with 'erp')
% 'limits', [... loerp hierp] puts duplicate "ghost" grey numbers on the ERP axis

% 
function [data,outsort,outtrials,limits,axhndls,erp,amps,cohers,cohsig] = boldimage(data,sortvar,times,titl,avewidth,decfactor,arg1,arg2,arg3,arg4,arg5,arg6,arg7,arg8,arg9)

%
%%%%%%%%%%%%%%%%%%%%%%% Define defaults %%%%%%%%%%%%%%%%%%%%%%%
%
YES = 1;  % logical variables
NO  = 0;

TIMEX = 1;          % 1 -> plot time on x-axis; 0 -> trials on x-axis
BACKCOLOR = [1 .984 .895]; % default grey background
PLOT_HEIGHT = 0.15; % fraction of y dim taken up by each time series axes
YGAP = 0.03;        % fraction gap between time axes
YEXPAND = 1.3;      % expansion factor for y-axis about erp, amp data limits
DEFAULT_AVEWIDTH =  1; % smooth trials with this window size by default
DEFAULT_DECFACTOR = 1; % decimate by this factor by default
DEFAULT_CYCLES = 3; % use this many cycles in amp,coher computation window
DEFAULT_CBAR = NO;  % do not plot color bar by default

LABELFONT = 14;     % font sizes for axis labels, tick labels
TICKFONT  = 12;
alpha     = 0.01;   % default alpha level for coherence significance
cohsig    = NO;     % default: do not compute coherence significance

Noshow    = NO;     % show sortvar by default
Nosort    = NO;     % sort on sortvar by default
Caxflag   = NO;     % use default caxis by default
Caxis     = [];
Coherflag = NO;     % don't compute or show amp,coher by default
Erpflag   = NO;     % don't show erp average by default
Alignflag = NO;     % don't align data to sortvar by default
Colorbar  = NO;     % if YES, plot a colorbar to right of erp image
Limitflag = NO;     % plot whole times range by default
coherfreq = nan;
srate     = nan;
aligntime = nan;
timelimits= nan;

minerp = nan;
maxerp = nan;
minamp = nan;
maxamp = nan;
mincoh = nan;
maxcoh = nan;
ax1=0;ax2=0;ax3=0;ax4=0;

%
%%%%%%%%%%%%%%%%%%% Test, fill in commandline args %%%%%%%%%%%%%%
%
if nargin < 2
  help erpimage
  return
end

framestot = size(data,1)*size(data,2);
ntrials = length(sortvar);
if ntrials < 2
  help erpimage
  fprintf('\nerpimage(): too few trials.\n');
  return
end

frames = floor(framestot/ntrials);
if frames*ntrials ~= framestot
  help erpimage
  fprintf(...
    '\nerpimage(); length of sortvar doesnt divide no. of data elements.\n')
  return
end

if nargin < 6
  decfactor = 0;
end
if nargin < 5
  avewidth = 0;
end
if nargin<4
  titl = '';
end
if nargin<3
  times = NO;
end
if times ==NO,
   times = NO:frames-1;
end

if avewidth == 0,
  avewidth = DEFAULT_AVEWIDTH;
end
if decfactor == 0,
  decfactor = DEFAULT_DECFACTOR;
end
if avewidth < 1
  help erpimage
  fprintf('\nerpimage(): Variable avewidth cannot be < 1.\n')
  return
end
if avewidth > ntrials
  fprintf('Setting variable avewidth to max %d.\n',ntrials)
  avewidth = ntrials;  
end
if decfactor < 1
  help erpimage
  fprintf('\nerpimage(): Variable decfactor cannot be < 1.\n')
  return
end
if decfactor > ntrials
  fprintf('Setting variable decfactor to max %d.\n',ntrials)
  decfactor = ntrials;  
end
%
%%%%%%%%%%%%%%%%% Collect option args %%%%%%%%%%%%%%%%%%%%%%%%%
%
if nargin > 6
  flagargs = [];
  for i=7:nargin
    Arg = eval(['arg' int2str(i-6)]);
    if Caxflag == YES
      Caxis = Arg;
      if size(Caxis,1) ~= 1 | size(Caxis,2) ~= 2
        help erpimage
        fprintf('\nerpimage(): caxis arg must be a (1,2) vector.\n');
        return
      end
      Caxflag = NO;
    elseif Coherflag == YES
       if size(Arg,1) ~= 1 | ( size(Arg,2) ~= 2 & size(Arg,2) ~= 3)
        help erpimage
        fprintf('\nerpimage(): coher arg must be a size (1,2) or (1,3) vector.\n');
        Arg
        return
       end
       coherfreq = Arg(1);
       srate     = Arg(2);
       if size(Arg,2) == 3
         cohsig = YES;
         alpha  = Arg(3);
         if alpha <= 0 | alpha > 0.1
           fprintf('erpimage(): alpha value %g out of bounds.\n',alpha); 
           return
         end
       end
       if coherfreq <= 0 | coherfreq > srate/2 | srate <= 0
        help erpimage
        fprintf('\nerpimage(): coher frequency (%g) out of range.\n',coherfreq);
        return
       end
       Coherflag = NO;
       Erpflag = YES;  % plot amp, coher below erp time series
    elseif Alignflag == YES
       if size(Arg,1) ~= 1 | size(Arg,2) ~= 2
        help erpimage
        fprintf('\nerpimage(): align arg must be a (1,2) vector.\n');
        return
       end
       aligntime = Arg(1);
       srate = Arg(2);
       if srate <= 0 
        help erpimage
        fprintf('\nerpimage(): align srate must be > 0.\n');
        return
       end
       Alignflag = NO;
    elseif Limitflag == YES
      %  [lotime hitime loerp hierp loamp hiamp locoher hicoher]
      if size(Arg,1) ~= 1 | size(Arg,2) < 2 ...
          | size(Arg,2) > 8 ...
        help erpimage
        fprintf('\nerpimage(): limits arg must be a vector sized (1,2<->8).\n');
        return
      end
      if  Arg(1) ~= nan & ...
          (Arg(2) <= Arg(1) ...
          | Arg(1) < min(times) ...
          | Arg(2) > max(times))
        help erpimage
        fprintf('\nerpimage(): time limits out of order or out of range.\n');
        return
      end
      timelimits = Arg(1:2);
      if length(Arg)> 2
        minerp = Arg(3);
      end
      if length(Arg)> 3
        maxerp = Arg(4);
      end
      if maxerp ~= nan & maxerp <= minerp
        help erpimage
        fprintf('\nerpimage(): erp limits args out of order.\n');
        return
      end
      if length(Arg)> 4
        minamp = Arg(5);
      end
      if length(Arg)> 5
        maxamp = Arg(6);
      end
      if maxamp <= minamp
        help erpimage
        fprintf('\nerpimage(): amp limits args out of order.\n');
        return
      end
      if length(Arg)> 6
        mincoh = Arg(7);
      end
      if length(Arg)> 7
        maxcoh = Arg(8);
      end
      if maxcoh <= mincoh
        help erpimage
        fprintf('\nerpimage(): coh limits args out of order.\n');
        return
      end
      Limitflag = NO;
    elseif strcmp(Arg,'nosort')
      Nosort = YES;
    elseif strcmp(Arg,'noplot')
      Noshow = YES;
    elseif strcmp(Arg,'caxis')
      Caxflag = YES;
    elseif strcmp(Arg,'coher')
      Coherflag = YES;
    elseif strcmp(Arg,'erp')
      Erpflag = YES;
    elseif strcmp(Arg,'align')
      Alignflag = YES;
    elseif strcmp(Arg,'cbar')| strcmp(Arg,'colorbar')
      Colorbar = YES;
    elseif strcmp(Arg,'limits')
      Limitflag = YES;
    else
      help erpimage
      fprintf('\nerpimage(): unknown arg %s\n',Arg);
      return
    end
  end
end

if   Caxflag == YES ...
  |Coherflag == YES ...
  |Alignflag == YES ...
  |Limitflag == YES
    help erpimage
    fprintf('\nerpimage(): missing option arg.\n')
    return
end
if isnan(timelimits)
   timelimits = [min(times) max(times)];
end
if ~isnan(aligntime)
 if aligntime ~= 0 ...
      & (aligntime < timelimits(1) | aligntime > timelimits(2))
  help erpimage
  fprintf('\nerpimage(): align time arg out of time limits.\n');
  return
 end
end
%
%%%%%%%%%%%%%% Reshape data to (frames,ntrials) %%%%%%%%%%%%%
%
data=reshape(data,frames,ntrials);
fprintf('Plotting input data as %d epochs of %d frames.\n',...
                             ntrials,frames);
%
%%%%%%%%%%%%%%% Sort the data on sortvar %%%%%%%%%%%%%%%%%%%%
%
if Nosort == YES
  fprintf('Not sorting data on input sortvar.\n');
else
  fprintf('Sorting data on input sortvar.\n');
  [sortvar,ix] = sort(sortvar);
  data = data(:,ix);
end
if max(sortvar)<0
   fprintf('Inverting sortvar to make it positive.\n');
   sortvar = -sortvar;
end
%
%%%%%%%%%%%%%%%%%%% Align data to sortvar %%%%%%%%%%%%%%%%%%%
%
if ~isnan(aligntime)
  if aligntime == 0
   ssv = sort(sortvar);
   aligntime= median(sortvar);
   % aligntime= median(ssv(round(ntrials/20)):round(19*ntrials/20)); 
                       % trimmed median - ignore top/bottom 5%
   fprintf('Aligning data to sortvar at %g ms.\n',aligntime);
  end
  aligndata=0*ones(frames,ntrials); % begin with matrix of nan's
  shifts = zeros(1,ntrials);
  for i=1:ntrials, %%%%%%%%% foreach trial %%%%%%%%%
   shft = round((aligntime-sortvar(i))*srate/1000);
   shifts(i) = shft;
   if shft>0, % shift right
    if frames-shft > 0
     aligndata(shft+1:frames,i)=data(1:frames-shft,i);
    else
     fprintf('No aligned data for epoch %d - shift too large.\n',i);
    end
   elseif shft < 0 % shift left
    if frames+shft > 0
     aligndata(1:frames+shft,i)=data(1-shft:frames,i);
    else
     fprintf('No aligned data for epoch %d - shift too large.\n',i);
    end
   else % shft == 0
     aligndata(:,i) = data(:,i);
   end 
  end % end trial
  fprintf('Shifted epochs by %d to %d frames.\n',min(shifts),max(shifts));
  data = aligndata;                       % now data is aligned to sortvar
  sortvar = zeros(1,ntrials)+aligntime;   % now sortvar is constant
end 
% 
%%%%%%%%%%%%%%%%%% Smooth data using moving average %%%%%%%%
%
if ~isnan(coherfreq)
  urdata = data; % compute amp, coher on unsmoothed data
end
if avewidth > 1 | decfactor > 1
  if Nosort == YES
    fprintf('Smoothing the data using a window width of %g epochs ',avewidth);
  else
    fprintf('Smoothing the sorted data using a window width of %g epochs ',...
                       avewidth);
  end
  fprintf('\n');
  fprintf('  and a decimation factor of %g\n',decfactor);
  [data,outtrials] = movav(data,1:ntrials,avewidth,decfactor); 
                                            % Note: using square window
  [outsort,outtrials] = movav(sortvar,1:ntrials,avewidth,decfactor); 
  fprintf('Output data will be %d frames by %d smoothed trials.\n',...
                          frames,length(outtrials));
else
  outtrials = 1:ntrials;
  outsort = sortvar;
end

%
%%%%%%%%%%%%%%%%%%%%%%%%% Find color axis limits %%%%%%%%%%%%%%%%%
%
if ~isempty(Caxis) 
  mindat = Caxis(1);
  maxdat = Caxis(2);
  fprintf('Using the specified caxis range of [%g,%g].\n',...
                                           mindat,maxdat);
else
  mindat = min(min(data));
  maxdat = max(max(data));
  maxdat =  max(abs([mindat maxdat])); % make symmetrical about 0
  mindat = -max(abs([mindat maxdat]));
  fprintf(...
     'The caxis range will be the sym. abs. data range [%g,%g].\n',...
                                                     mindat,maxdat);
end
%
%%%%%%%%%%%%%%%%%%%%%%%%%% Set time limits %%%%%%%%%%%%%%%%%%%%%%
%
if ~isnan(timelimits(1))
    timelimits = [min(times) max(times)];
end
%
%%%%%%%%%%%%% Image the aligned/sorted/smoothed data %%%%%%%%%%%%
%
if ~isnan(coherfreq)       % if plot three time axes
     image_loy = 3*PLOT_HEIGHT;
elseif Erpflag == YES   % elseif if plot only one time axes
     image_loy = 1*PLOT_HEIGHT;
else                    % else plot erp-image only
     image_loy = 0*PLOT_HEIGHT;
end
pos=get(gca,'Position');
delete(gca)
ax1=axes('Position',...
    [pos(1) pos(2)+image_loy*pos(4) ...
     pos(3) (1-image_loy)*pos(4)]);

  ind = isnan(data);    % find nan's in data
  [i j]=find(ind==1);
  if ~isempty(i)
    data(i,j) = 0;      % plot shifted nan data as 0 (=green)
  end
if TIMEX
  imagesc(times,outtrials,data',[mindat,maxdat]);% plot time on x-axis
  set(gca,'Ydir','normal');
  pp = get(gca,'position');
  set(gca,'position',[pp(1)+pp(3)*0.05 pp(2) pp(3)*0.95 pp(4)]);
%  set(gca, 'YTickLabel', [1 2 3 4 5 6 7 8 9 10]);
  axis([timelimits(1) timelimits(2) ...
       min(outtrials) max(outtrials)]);
else
  imagesc(outtrials,times,data,[mindat,maxdat]); % plot trials on x-axis
  axis([min(outtrials) max(outtrials)...
       timelimits(1) timelimits(2)]);
end
p1 = get(gca,'position');
p1(3) = 0.95*p1(3);
set(gca,'FontSize',TICKFONT,...
	'position',p1);
	%'fontweight','bold');
hold on;
if times(1) < 0 & times(frames) > 0
  plot([0 0],[0 ntrials],'k:','Linewidth',1.9); % plot vertical line at time 0
end

if Noshow == NO & (min(outsort) < timelimits(1) ...
   | max(outsort) > timelimits(2))
  fprintf('Not all sortvar values within time vector limits: \n')
  fprintf('        outliers will be shown at nearest limit.\n');
  i = find(outsort< timelimits(1));
  outsort(i) = timelimits(1);
  i = find(outsort> timelimits(2));
  outsort(i) = timelimits(2);
end

if TIMEX
 if Nosort == YES
  l=ylabel('Trial Number');
 else
  l=ylabel('RT-Sorted Trial Index');
 end
else
 if Nosort == YES
  l=xlabel('Trial Number');
 else
  l=xlabel('RT-Sorted Trial Index');
 end
end
set(l,'FontSize',LABELFONT);

t=title(titl);
set(t,'FontSize',LABELFONT);

set(gca,'Box','off');
set(gca,'Fontsize',TICKFONT);
set(gca,'color',BACKCOLOR);
p1 = get(get(gca,'ylabel'),'position');
if Erpflag == NO
  l=xlabel('Time (ms)');
  set(l,'Fontsize',LABELFONT);
end
%
%%%%%%%%%%%%%%%%%%%% Overplot sortvar %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
if Noshow == YES
  fprintf('Not overplotting sorted sortvar on data.\n');
else
  fprintf('Overplotting sorted sortvar on data.\n');
  e = 0:ntrials-1;
  hold on; 
  if TIMEX      % overplot sortvar
    plot(outsort,outtrials,'k','LineWidth',2); hold off 
  else
    plot(outtrials,outsort,'k','LineWidth',2); hold off
  end                                                 
end
%
%%%%%%%%%%%%%%%%%%%%%%%% Plot colorbar %%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
if Colorbar == YES
   cbar % plot colorbar to right of image
   set(gca,'FontSize',TICKFONT,...
	   'fontweight','normal');
end
%
%%%%%%%%%%%%%%%%%%%%%%% Compute ERP %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
if Erpflag == YES
 set(gca,'Xticklabel',[]);    % remove tick labels from bottom of image
 erp=nanmean(data');          % compute erp average, ignoring nan's
 %
 %%%%%% Plot ERP time series below image %%%%%%%%%%%%%%%%%
 %
 if isnan(maxerp)
  fac = 10;
  maxerp = 0;
  while maxerp == 0
   maxerp = round(fac*YEXPAND*max(erp))/fac; % minimal decimal places
   fac = 10*fac;
  end
 end
 if isnan(minerp)
  fac = 1;
  minerp = 0;
  while minerp == 0
    minerp = round(fac*YEXPAND*min(erp))/fac; % minimal decimal places
    fac = 10*fac;
  end
 end
 %limit = [timelimits(1:2) -max(abs([minerp maxerp])) max(abs([minerp maxerp]))];
% if maxerp>=0,
%   limit = [timelimits(1:2) 0 maxerp];
% else
%   limit = [timelimits(1:2) maxerp 0];
% end
 dd = max(erp)-min(erp);
 limit = [timelimits(1:2) min(erp)-0.1*dd max(erp)+0.1*dd];
 if ~isnan(coherfreq)
  ax2=axes('Position',...
     [pos(1)+0.05*pos(3) pos(2)+2/3*image_loy*pos(4) ...
      0.95*pos(3) (image_loy/3-YGAP)*pos(4)]);
 else
  ax2=axes('Position',...
     [pos(1)+0.05*pos(3) pos(2) ...
      0.95*pos(3) image_loy*pos(4)]);
 end
 
 %plot1erp(ax2,times,erp,limit);
 plot(times,erp,'LineWidth',4);
 axis(limit);
 set(ax2,'Box','off','linewidth',2); 
 set(ax2,'yaxislocation','right');
 set(ax2,'ycolor',BACKCOLOR);
 line([limit(1:2)],[0 0],'Color','k','linewidth',3);
 line([0 0],[limit(3:4)],'Color','k','linewidth',3);

% set(ax2,'YColor',BACKCOLOR);
 if isnan(coherfreq)
  if TIMEX
   l=xlabel('Time (s)');
   set(l,'FontSize',LABELFONT);
  else
   l=ylabel('Time (s)');
   set(l,'FontSize',LABELFONT);
  end
 end

 ydelta = 1/10*(times(frames)-times(1));
 ytextoffset = times(1)-0.8*ydelta;
 ynumoffset = times(1)+0.3*ydelta;

% t=text(ynumoffset,0.7*limit(3), num2str(limit(3)));
% set(t,'HorizontalAlignment','right','FontSize',TICKFONT)

% t=text(ynumoffset,0.7*limit(4), num2str(limit(4)));
% set(t,'HorizontalAlignment','right','FontSize',TICKFONT)

 ynum=0.7*(limit(3)+limit(4))/2 - 0.5;
 t=text(ytextoffset,limit(3),'\Delta BOLD (%)','Rotation',90);
 set(t,'HorizontalAlignment','left',...
       'FontSize',LABELFONT);

 set(ax2,'Fontsize',TICKFONT,...
	 'fontweight','normal');
 set(gca,'Box','off','color',BACKCOLOR);
else
  erp = [];
end
%
%%%%%%%%%%%%%%%%%%%%% Plot amp, coher time series %%%%%%%%%%%%%%
%
if ~isnan(coherfreq) 
   %
   %%%%%% Plot amp axis below ERP %%%%%%%%%%%%%%%%%%%%%%%%
   %
   axis('off')
   if cohsig == NO
     [amps,cohers] = ...
       phasecoher(urdata,size(times,2),srate,coherfreq,DEFAULT_CYCLES);
   else
     fprintf('Computing and plotting %g coherence significance level...\n',alpha);
     [amps,cohers,cohsig] = ...
       phasecoher(urdata,size(times,2),srate,coherfreq,DEFAULT_CYCLES,alpha);
     fprintf('Coherence significance level: %g\n',cohsig);
   end
   ax3=axes('Position',...
     [pos(1) pos(2)+1/3*image_loy*pos(4) ...
      pos(3) (image_loy/3-YGAP)*pos(4)]);
   if isnan(maxamp)
    fac = 1;
    maxamp = 0;
    while maxamp == 0
     maxamp = floor(YEXPAND*fac*max(amps))/fac; % minimal decimal place
     fac = 10*fac;
    end
    maxamp = maxamp + 10/fac;
   end
   if isnan(minamp)
     minamp = 0;
   end
   plot1erp(ax3,times,amps,[timelimits(1) timelimits(2) minamp maxamp])
   set(ax3,'YColor',BACKCOLOR);
   axis('off')
   t=text(ynumoffset,0, num2str(0));
   set(t,'HorizontalAlignment','right','FontSize',TICKFONT)

   t=text(ynumoffset,maxamp, num2str(maxamp));
   set(t,'HorizontalAlignment','right','FontSize',TICKFONT)

   t=text(ytextoffset,maxamp/2,'Rms','Rotation',90);
   set(t,'HorizontalAlignment','center','FontSize',LABELFONT)
   %
   %%%%%% Make coher axis below amp %%%%%%%%%%%%%%%%%%%%%%
   %
   ax4=axes('Position',...
     [pos(1) pos(2) ...
      pos(3) (image_loy/3-YGAP)*pos(4)]);
   if isnan(maxcoh)
    fac = 1;
    maxcoh = 0;
    while maxcoh == 0
     maxcoh = floor(YEXPAND*fac*max(cohers))/fac; % minimal decimal place
     fac = 10*fac;
    end
    maxcoh = maxcoh + 10/fac;
   end
   if isnan(mincoh)
     mincoh = 0;
   end
   plot1erp(ax4,times,cohers,[limit(1:2) mincoh maxcoh])
%   set(ax4,'XColor',BACKCOLOR);
   set(ax4,'YColor',BACKCOLOR);

   t=text(ynumoffset,0, num2str(0));
   set(t,'HorizontalAlignment','right','FontSize',TICKFONT)

   t=text(ynumoffset,maxcoh, num2str(maxcoh));
   set(t,'HorizontalAlignment','right','FontSize',TICKFONT)

   t=text(ytextoffset,maxcoh/2,'Coh','Rotation',90);
   set(t,'HorizontalAlignment','center','FontSize',LABELFONT)

   if cohsig > 0   % plot coherence significance level
     hold on
     plot([timelimits(1) timelimits(2)],[cohsig cohsig],'r');
   end

   set(gca,'Box','off','color',BACKCOLOR);
   set(gca,'Fontsize',TICKFONT);
%   l=xlabel('Time (ms)');
   set(l,'Fontsize',LABELFONT);
   axtmp = axis;
   text(8/13*(axtmp(2)-axtmp(1))+axtmp(1), ...
        8/13*(axtmp(4)-axtmp(3))+axtmp(3), ...
        [num2str(coherfreq,4) ' Hz']);
else
   amps   = [];    % null outputs unless coherfreq specified
   cohers = [];
end
limits = [timelimits(1:2) minerp maxerp minamp maxamp mincoh maxcoh];
axhndls = [ax1 ax2 ax3 ax4];

%
%%%%%%%%%%%%%%%%%%% function plot1erp() %%%%%%%%%%%%%%%%%%%%%%%%
%
function plot1erp(ax,Time,erp,axlimits)

%  plot(Time,erp,'LineWidth',1.9); hold on
%  axis([axlimits(1:2) 1.1*axlimits(3:4)])
%  line([axlimits(1:2)],[0 0],'Color','k'); % y=zero-line
%  line([0 0],[axlimits(3:4)*1.1],'Color','k'); % x=zero-line





