% export_result() - convert both the selected ROAs (component maps) 
%                   and structural images to ANALYZE format. The 
%                   resulting files include *.img as well as the
%                   header file *.hdr could be used in Spatial 
%                   Normalization routine to warp the map to
%                   standard brain for 3D rendering on the SPM99 
%                   brain template.
% 
%                   This routine can also be used in stanlone mode 
%                   once you load up the FMRI structural data set 
%                   (xxxx.fmr) under MATLAB environment. 
%
% Usage:
%   >> export_result;
%
% Inputs:
%
% Outputs:
%
% Author: Jeng-Ren Duann, CNL/Salk Inst. & SCCN/INC/UCSD, 2001-10
%
% See also:fmrlab_normalize(), fmrlab_render(), show_2d(),
%          show_3d()

% --
% Copyright (C) 2002 Jeng-Ren Duann, Salk Institute, duann@salk.edu
%
% This program is free software; you can redistribute it and/or
% modify it under the terms of the GNU General Public License as
% published by the Free Software Foundation; either version 2 of
% the License, or (at your option) any later version.
%
% This program is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU General Public License for more details.
%
% You should have received a copy of the GNU General Public License
% along with this program; if not, write to the Free Software
% Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

% $Log: export_result.m,v $
% Revision 1.2  2003/02/13 00:24:13  duann
% FMRI.struc_flip and FMRI.struc_resort fields missing. Add them.
%
% Revision 1.1  2003/02/06 18:57:54  duann
% Initial revision
%

function export_result()
  
  global FMRI
  global hwnd

  FMRI.struc_flip = 1;
  FMRI.struc_resort = 0;
  
  if length(FMRI) == 0,
    fmrlab_messagedlg('Export Selected ROAs',...
	       'Error:',...
	       ['No current data set exists. Please load a ''.fmr''' ...
		' data files first']);
    return;
  end
    
  % check if there is structural image data set in FMRI structure
  no_struc = 0;
  if isempty(FMRI.struc_name),
    str = ['No structural images specified. Spatial normalization' ...
	   [' will not be possible. Save ROAs in ANALYZE format' ...
	    ' only?']];
    button_pressed=fmrlab_messagedlg('Export Selected ROAs',...
		  'Warning:',...
		  str,{'Yes', 'No'});
    if  strcmp(button_pressed,'No')==1,
      fmrlab_messagedlg('Export Selected ROAs',...
		    'Warning:',...
		    'See  Dataset > Edit Struct. Info');
      return;
    else
      no_struc = 1;
    end
  end
  
  % check if FMRI.icaweight is empty
  if isempty(FMRI.icaweight),
    str = ['No ICA weight matrix. Will export structural images' ...
	   [' only (in ANALYZE format). To export component ROA' ...
	    ' map(s) as well, see: Process > Run ICA']];
    button_pressed = fmrlab_messagedlg('Export Selected ROAs',...
		  'Warning:',...
		  str,{'Continue','Cancel'});
    if strcmp(button_pressed,'Cancel') == 1,
      return;
    end
  end
  
  % check if no ICA component marked
  if isempty(FMRI.icacomp),
    str = ['No ICA components marked for export. Will export' ...
	   [' structural image only (in ANALYZE format). To export' ...
	    [' component ROA map(s) as well, see: Visualize > Component' ...
	     ' ROAs']]];
    button_pressed = fmrlab_messagedlg('Export Selected ROAs',...
				    'Warning:',...
				    str,...
				    {'Continue','Cancel'});
    if strcmp(button_pressed,'Cancel') == 1,
      return;
    end
  end
  
  % ask user if he/she wants to apply flip/reorder image slices
  str = sprintf('If ''right is left'' in data:');
  str1 = sprintf('%s\nFlip images (left<->right)',str);
  str = sprintf('If ''first slice is superior'' in data:');
  str2 = sprintf('%s\nReorder slices (last<->first)',str);
  prompt = {str1, str2};
  a = fmrlab_inputdlg(prompt,...
		  'Export Selected ROAs',...
		  2,...
		  {'yes','no'});
  if isempty(a),
    return;
  end
  if strcmp(lower(char(a(1))),'yes')==1,
    FMRI.struc_flip = 1;
  else
    FMRI.struc_flip = 0;
  end
  if strcmp(lower(char(a(2))),'no') == 1,
    FMRI.struc_reorder = 0;
  else
    FMRI.struc_reorder = 1;
  end
  
  % converting structural image if exists
  if ~isempty(FMRI.struc_name),
    disp('Converting structural images...');
    fname = [FMRI.struc_path FMRI.struc_name];
    fp = fopen(fname,'rb','ieee-be');
    data = fread(fp,'uint16');
    fclose(fp);
    % flip image if necessary
    target_data = [];
    for i=1:FMRI.struc_image_depth,
      tempdata = data((i-1)*FMRI.struc_image_width*...
			FMRI.struc_image_height+1:...
			i*FMRI.struc_image_width*...
			FMRI.struc_image_height);
      if no_struc ~= 1,
        if FMRI.struc_flip == 1,
	  tempdata = reshape(...
	      tempdata,...
	      FMRI.struc_image_width,...
	      FMRI.struc_image_height);
	  tempdata = flipud(tempdata);
	  tempdata = reshape(tempdata,1,...
			     FMRI.struc_image_width*...
			     FMRI.struc_image_height);
        end
        if FMRI.struc_reorder == 1,
	  target_data = [tempdata target_data];
        else
	  target_data = [target_data tempdata];
        end
      else
	target_data = [target_data tempdata];
      end
    end
    
    hdr.data = target_data;
    hdr.datatype = 4; % for int16
    hdr.imagewidth = FMRI.struc_image_width;
    hdr.imageheight = FMRI.struc_image_height;
    hdr.imagedepth = FMRI.struc_image_depth;
    hdr.volsizex = FMRI.struc_fov_x / FMRI.struc_image_width;
    hdr.volsizey = FMRI.struc_fov_y / FMRI.struc_image_height;
    hdr.volsizez = FMRI.struc_slicetk;
    write_analyze('str_0001',hdr);
  end
  
  % write ROAs
  if length(FMRI.icacomp) ~= 0,
    W = FMRI.icaweight * FMRI.icasphere;
    u = W * FMRI.data;
    for i=1:length(FMRI.icacomp),
      tempdata = u(FMRI.icacomp(i), :);
      mean_tempdata = mean(tempdata);
      std_tempdata = std(tempdata);
      tempdata = (tempdata-mean_tempdata)/std_tempdata;
      % thresholding z map
      ii = find(abs(tempdata)<FMRI.ica_zthre);
      tempdata(ii) = 0.0;
      ii = find(abs(tempdata) > FMRI.ica_zbound);
      tempdata(ii) = FMRI.ica_zbound;
      data = zeros(FMRI.dime_x*FMRI.dime_y*FMRI.dime_z,1);
      for j=1:length(FMRI.idx),
	data(FMRI.idx(j)) = tempdata(j);
      end
      target_data = [];
      for j=1:FMRI.dime_z,
	tempdata = data((j-1)*FMRI.dime_x*...
			FMRI.dime_y+1:...
			j*FMRI.dime_x*...
			FMRI.dime_y);
	
	if no_struc ~= 1,
	  if FMRI.struc_flip == 1,
	    tempdata = reshape(...
		tempdata,FMRI.dime_x,FMRI.dime_y);
	    tempdata = flipud(tempdata);
	    tempdata = reshape(tempdata,...
			       1,...
			       FMRI.dime_x*FMRI.dime_y);
	  end
	  if FMRI.struc_reorder == 1,
	    target_data = [tempdata target_data];
	  else
	    target_data = [target_data tempdata];
	  end
	else
	  target_data = [target_data tempdata];
	end
      end
    
      hdr.data = target_data;
      hdr.datatype = 16; % for float
      hdr.imagewidth = FMRI.dime_x;
      hdr.imageheight = FMRI.dime_y;
      hdr.imagedepth = FMRI.dime_z;
      hdr.volsizex = FMRI.fov_x / FMRI.dime_x;
      hdr.volsizey = FMRI.fov_y / FMRI.dime_y;
      hdr.volsizez = FMRI.slicetk;
      ftemp = sprintf('roa_%04d',FMRI.icacomp(i));
      write_analyze(ftemp,hdr);
      
    end
  end
  
  disp('Done.');
  
  return;
  
%========================================
function write_analyze(file_temp,content)
%========================================
  global FMRI
  
  disp(['Writing ' file_temp '.img ...']);
  fname = [file_temp '.img'];
  fp = fopen(fname,'wb','ieee-be');
  if fp == -1,
    %errordlg('Cannot open image file ...!');
    fmrlab_messagedlg('Export Results',...
		  'Error:',...
		  'Cannot open image file');
    return;
  end
  switch content.datatype,
   case 4,
    fwrite(fp,content.data,'int16');
   case 8,
    fwrite(fp,content.data,'int32');
   case 16,
    fwrite(fp,content.data,'float');
   case 32,
    fwrite(fp,content.data,'double');
  end
  fclose(fp);
  
  disp(['Writing ' file_temp '.hdr ...']);
  fname = [file_temp '.hdr'];
  fp = fopen(fname, 'wb','ieee-be');
  if fp == -1,
    %errordlg('Cannot open header file...!');
    fmrlab_messagedlg('Export Results',...
		  'Error:',...
		  'Cannot open header file');
    return;
  end
  if length(FMRI.struc_path) < 18,
    db_name = FMRI.struc_path;
    while length(db_name) < 18,
      db_name = [db_name 0];
    end
  else
    ss = length(FMRI.struc_path)-18;
    ee = length(FMRI.struc_path)-1;
    db_name = FMRI.struc_path(ss:ee);
  end  
  data_type = ['dsr      ' 0];
  DESCRIP = 'spm compatible';
  descrip = zeros(1, 80);
  d = 1:length(DESCRIP);
  descrip(d) = DESCRIP(d);
  aux_file = ['none                   ' 0];
  origin = [0 0 0 0 0];
  if content.datatype == 4,
    bitpix = 16;
  else
    bitpix = 32;
  end
  glmax = 32767;
  glmin = 0;
  vox_offset = 0;
  funused1 = 1; %Scale factor
  dim = [4 content.imagewidth content.imageheight ...
	 content.imagedepth 1 0 0 0];
  pixdim = [0 content.volsizex content.volsizey content.volsizez 0 0 0 0];
  orientation = char(3);
  
  % write (struct) header_key
  %--------------------------------------------------------------------
  fwrite(fp, 348, 'int32');		%sizeof_hdr
  fwrite(fp, data_type, 'char');	%data_type
  fwrite(fp, db_name, 'char');	%db_name
  fwrite(fp, 0, 'int32');		%extents
  fwrite(fp, 0, 'int16');		%session_error
  fwrite(fp, 'r', 'char');		%regular
  fwrite(fp, '0', 'char');		%hkey_un0
  % write (struct) image_dimension
  %--------------------------------------------------------------------
  fwrite(fp, dim, 'int16');		%short int dim[8]
  fwrite(fp, 'mm', 'char');		%char vox_units[4]
  fwrite(fp, 0, 'char');
  fwrite(fp, 0, 'char');
  fwrite(fp, zeros(1, 8), 'char');  %char cal_units[8]
  fwrite(fp, 0, 'int16');		%short int unused1
  fwrite(fp, content.datatype, 'int16');	%short int datatype
  fwrite(fp, bitpix, 'int16');	%short int bitpix
  fwrite(fp, 0, 'int16');		%short int dim_un0
  fwrite(fp, pixdim, 'float');	%float pixdim[8]
  fwrite(fp, vox_offset, 'float');  %float vox_offset
  fwrite(fp, funused1, 'float');	%float funused1 (scale factor)
  fwrite(fp, 0, 'float');		%float funused2
  fwrite(fp, 0, 'float');		%float funused3
  fwrite(fp, 0, 'float');		%float cal_max
  fwrite(fp, 0, 'float');		%float cal_min
  fwrite(fp, 0, 'int32');		%int compressed
  fwrite(fp, 0, 'int32');		%int verified
  fwrite(fp,glmax,	'int32');	%int glmax
  fwrite(fp,glmin,	'int32');	%int glmin
  %write (struct) data_history
  %-------------------------------------------------------------------
  fwrite(fp, descrip, 'char');	%char descrip[80]
  fwrite(fp, aux_file, 'char');	%char aux_file[24]
  fwrite(fp, orientation, 'char');	%char orient
  fwrite(fp, origin, 'int16');	%short int origin[5]
  fwrite(fp, zeros(1, 85), 'char');
  %-------------------------------------------------------------------
  fclose(fp);
  
  return;
