% map_on_struc() - component browser by overlaying ROA onto 2D
%                  slices of structural images with interactive 
%                  graphic user interface
%
% Usage:
%   >> map_on_struc(action)
%
% Inputs:
%   action - specifying action for the browse window to perform,
%            the value can be:
%          - 'initialize': initialize the graphic user interface
%          - 'prev_component': move to the previous coponent
%          - 'next_component': move to the next component
%          - 'goto_comp': goto the specified component indicated in
%            the text editing box
%          - 'add_list': add current component to the selection list
%          - 'goto_list': goto component highlighted in the
%            selection list
%          - 'remove_list': remove component from the selection list
%          - 'add_rej_list': add component into rejection list
%          - 'goto_rej_list': goto the component highlighted in the
%            rejection list
%          - 'rm_rej_list': remove component from the rejection list
%          - 'refresh': update the current view due to the change
%             of component selection or other actions
% 
% See also: map_on_fmri(), boldimage(), cbar()
%
% Notes: - map_on_struc() might need interpolation to make the ROA
%          the same coordinates with structural images. Thus, it might take
%          longer to browse component maps. It's highly recommended to use
%          map_on_fmri() first to add component of interest into the
%          selection list, then to use map_on_struc() to browse the selected
%          component only
% 
% --
% Copyright (C) 2002 Jeng-Ren Duann, Salk Institute, duann@salk.edu
%
% This program is free software; you can redistribute it and/or
% modify it under the terms of the GNU General Public License as
% published by the Free Software Foundation; either version 2 of
% the License, or (at your option) any later version.
%
% This program is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU General Public License for more details.
%
% You should have received a copy of the GNU General Public License
% along with this program; if not, write to the Free Software
% Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

function map_on_struc(action)

global FMRI
global BASEIMG
global struc_image_width
global struc_image_height

if length(FMRI)==0,
  fmrlab_messagedlg('Component ROAs on Struct. Images',...
		 'Error:',...
		 ['No current data set exists. Please load a ''.fmr''' ...
		  ' data file first']);
  return;
end

RGB_mainwin = [1 .984 .895];
RGB_background = [0.860 0.824 0.618];
RGB_button = [0.860 0.824 0.618];
RGB_text = [0.333 0.283 0.003];

if nargin<1,
  action = 'initialize';
end

if isempty(FMRI.icaweight),
  str = 'No ICA weight matrix. See:';
  str = [str sprintf('\nProcess > Run ICA')];
  fmrlab_messagedlg('Component ROAs on Struct. Images',...
		 'Error:',...
		 str);
  return;
end

if isempty(FMRI.struc_name),
  str = sprintf('No Structural images available.');
  str = [str sprintf(' See:\nDataset > Edit Struct. Info')];
  fmrlab_messagedlg('Component ROAs on Struct. Images',...
		 'Error:',...
		 str);
  return;
end

if strcmp(action, 'initialize'),
  % finding the component maps
  
  if isunix,
      text_size = 14;
  else
      text_size = 12;
  end
  
  progressbar(0.1);
  drawnow;
  
  W = FMRI.icaweight * FMRI.icasphere;
  progressbar(0.2);drawnow;

  invW = pinv(W);
  progressbar(0.4); drawnow;
  
  u = W * FMRI.data;
  progressbar(0.8);
  drawnow;

  figNumber = figure( ...
    'name', 'Display Component ROA on Structural Images', ...
    'numbertitle','off',...
    'doublebuffer', 'on', ...
    'color',RGB_mainwin,...
    'backingstore','off', ...
      'menubar','none');

  set(figNumber,'pointer','watch');
  
  pos = get(figNumber,'position');
  set(figNumber,'position',[5 260 700 500]);

  component_displayed = 1;
  component_selected = [];
  component_rejected = [];
  total_component = size(FMRI.icaweight,1);

  % use the training data set and the index of inside-brain voxels to 
  % construct the functional images used to overlay the component
  % map
  fname = [FMRI.struc_path FMRI.struc_name];
  fp = fopen(fname,'rb','ieee-be');
  avg_image = fread(fp,'uint16');
  fclose(fp);
  struc_image_width = FMRI.struc_image_width;
  struc_image_height = FMRI.struc_image_height;
  struc_slice_no = FMRI.struc_image_depth;
  
  base_image = zeros(1,struc_image_width*struc_image_height*struc_slice_no);
%  base_image = 64-63*(max(avg_image)-avg_image)/(max(avg_image)-min(avg_image));
  base_image = fmrlab_scale(avg_image,1,64);
  BASEIMG = base_image;
  base_image = reshape(base_image,struc_image_width, ...
		       struc_image_height,struc_slice_no);
  
  disp_col = ceil(sqrt(FMRI.dime_z));
  disp_row = ceil(FMRI.dime_z/disp_col);
%  if (disp_row == disp_col)
%    disp_row = disp_row + 1; 
%  end
  if mod(FMRI.dime_z,disp_col) == 0,
    disp_row = disp_row + 1;
  end  

  progressbar(0.9);
  drawnow
  figure(figNumber);

  ih = []; % create image hendle for each image patch
  for i=1:FMRI.dime_z,
    a = floor(i/disp_col);
    b = mod(i,disp_col);
    ii = tightsubplot(disp_row,disp_col,a*disp_col+b);
    image(base_image(:,:,i)');
    axis off
    axis image
    ih = [ih ii];
  end
  cb = linspace(65,128,64);
  cb = cb';
  b = repmat(cb,1,5);
  ii = tightsubplot(disp_row,disp_col,FMRI.dime_z+1);
  image(flipud(b));
  axis image
  set(ii,'position',[0.96 0.15 0.03 0.3]);
  set(ii,'xtick',[]);
  set(ii,'ytick',linspace(1,64,10));
  cb = linspace(8.0,1.5,5);
  cb = [cb linspace(-1.5,-8.0,5)];
  cb = num2str(cb','%4.1f');
  i = [2:4 7:9];
  for j=1:size(cb,2),
    cb(i,j) = ' ';
  end
  set(ii,'yticklabel',cb);
  fmrlab_color(2);
  ih = [ih ii]; % ih(FMRI.dime_z+1) = handle of colorbar

  ii = textsc('Component Map','title');
  set(ii,'fontsize',text_size+2');
  ih = [ih ii]; % ih(FMRI.dime_z+2) = handle of title

  f1 = uicontrol('style','frame',...
    'backgroundcolor',RGB_background);
  pos = get(f1,'position');
  set(f1,'position',[pos(1)-5 pos(2)-15 170 pos(4)+15]);
  pos = get(f1,'position');
  uicontrol('style','text',...
    'position',[pos(1) pos(4)+10 65 15],...
    'string','Browse',...
    'backgroundcolor',RGB_mainwin,...
    'fontsize',text_size);
  f2 = uicontrol('style','frame',...
    'backgroundcolor',RGB_background,...
    'position',[pos(1)+pos(3) pos(2) 230 pos(4)]);
  pos = get(f2,'position');
  uicontrol('style','text',...
    'position',[pos(1) pos(4)+10 150 15],...
    'string','Select Components',...
	    'horizontalalign','left',...
    'backgroundcolor',RGB_mainwin,...
    'fontsize',text_size);
  f4 = uicontrol('style','frame',...
		 'backgroundcolor',RGB_background,...
		 'position',[pos(1)+pos(3) pos(2) 230 pos(4)]);
  pos = get(f4,'position');
  uicontrol('style','text',...
	    'position',[pos(1) pos(4)+10 160 15],...
	    'string','Reject Components',...
	    'horizontalalign','left',...
	    'backgroundcolor',RGB_mainwin,...
	    'fontsize',text_size);
  f3 = uicontrol('style','frame',...
    'backgroundcolor',RGB_background,...
    'position',[pos(1)+pos(3) pos(2) 132 pos(4)]);
  pos = get(f3,'position');
  uicontrol('style','text',...
    'position',[pos(1) pos(4)+10 120 15],...
    'string','Time Course',...
	    'horizontalalign','left',...
    'backgroundcolor',RGB_mainwin,...
    'fontsize',text_size);

  callbackStr = 'map_on_struc(''prev_component'');';
  bt0 = uicontrol('style',    'pushbutton', ...
                  'string',   '<', ...
                  'fontsize',text_size, ...
                  'tooltipstring', 'View previous component map',...
                  'callback',callbackStr,...
		  'backgroundcolor',RGB_button,...
		  'foregroundcolor',RGB_text);
  pos = get(bt0,'position');
  set(bt0,'position',[pos(1) pos(2)-10 50 25]);

  pos = get(bt0,'position');
  callbackStr = 'map_on_struc(''next_component'');';
  bt1 = uicontrol('style',     'pushbutton', ...
                  'string',    '>', ...
                  'fontsize',text_size, ...
                  'position',  [pos(1)+pos(3)+5 pos(2) 50 25], ...
                  'tooltipstring', 'View next component map',...
                  'callback',  callbackStr,...
		  'backgroundcolor',RGB_button,...
		  'foregroundcolor',RGB_text);
  
  pos = get(bt1,'position');
  callbackStr = 'map_on_struc(''goto_comp'');';
  bt2 = uicontrol('style', 'edit', ...
    'string','1',...
    'fontsize',text_size, ...
    'tag','ed_gotocomp',...
    'backgroundcolor',RGB_mainwin,...
		  'foregroundcolor',RGB_text,...
    'position', [pos(1)+pos(3)+5 pos(2) 50 25], ...
    'tooltipstring', 'Display the given component number',...
    'callback', callbackStr);

  pos = get(bt2,'position');
  callbackStr = 'map_on_struc(''add_list'');';
  bt3 = uicontrol('style', 'pushbutton', ...
    'string',   'Mark', ...
    'fontsize',text_size, ...
    'position', [pos(1)+pos(3)+10 pos(2) 70 25], ...
    'tooltipstring', 'Push current component map into list',...
    'callback', callbackStr,...
		  'backgroundcolor',RGB_button,...
		  'foregroundcolor',RGB_text);
  
  pos = get(bt3,'position');
  if isempty(FMRI.icacomp),
    FMRI.icacomp = [];
  end
  if length(FMRI.icacomp) == 0,
    listStr = '0';
  else
    listStr = int2str(FMRI.icacomp);
    component_selected = FMRI.icacomp;
  end
  callbackStr = 'map_on_struc(''goto_list'');';
  list1 = uicontrol('style', 'popupmenu', ...
    'tag','comp_select',...
    'fontsize',text_size, ...
    'string', listStr, ...
    'callback', callbackStr, ...
    'backgroundcolor', RGB_background,...
		    'foregroundcolor',RGB_text,...
    'tooltipstring', 'Select a component map to display from list',...
    'position', [pos(1)+pos(3)+5 pos(2) 70 25]);

  pos = get(list1,'position');
  callbackStr = 'map_on_struc(''remove_list'');';
  bt4 = uicontrol('style', 'pushbutton', ...
    'string',   'Unmark', ...
    'fontsize',text_size, ...
    'position', [pos(1)+pos(3)+5 pos(2) 70 25], ...
    'tooltipstring', 'Remove current component map from list',...
    'callback', callbackStr,...
		  'backgroundcolor',RGB_button,...
		  'foregroundcolor',RGB_text);

  pos = get(f4,'position');
  callbackstr = 'map_on_struc(''add_rej_list'')';
  bt5 = uicontrol('style','pushbutton',...
		  'string','Mark',...
		  'fontsize',text_size,...
		  'position',[pos(1)+5 pos(2)+5 70 25],...
		  'tooltipstring','Add current component to reject list',...
		  'callback',callbackstr,...
		  'backgroundcolor',RGB_button,...
		  'foregroundcolor',RGB_text);
  if length(FMRI.rejcomp) == 0,
    rejstr = '0';
  else
    rejstr = int2str(FMRI.rejcomp);
    component_rejected = FMRI.rejcomp;
  end
  callbackstr = 'map_on_struc(''goto_rej_list'')';
  list2 = uicontrol('style','popupmenu',...
		    'tag','rej_list',...
		    'fontsize',text_size,...
		    'string',rejstr,...
		    'callback',callbackstr,...
		    'backgroundcolor',RGB_background,...
		    'foregroundcolor',RGB_text,...
		    'tooltipstring',['Select a component map to' ...
		    ' display from list'],...
		    'position',[pos(1)+80 pos(2)+5 70 25]);
  callbackstr = 'map_on_struc(''rm_rej_list'')';
  bt6 = uicontrol('style','pushbutton',...
		  'string','Unmark',...
		  'fontsize',text_size,...
		  'position',[pos(1)+155 pos(2)+5 70 25],...
		  'tooltipstring','Remove component from rejected list',...
		  'callback',callbackstr,...
		  'backgroundcolor',RGB_button,...
		  'foregroundcolor',RGB_text);
  
  pos = get(f3,'position');
  callbackStr = 'map_on_struc(''refresh'');';
  z = {'None', 'Comp.', 'ROA', 'Comp+ROA', 'BOLD Image'};
  check1 = uicontrol('style','popupmenu',...
    'tag','time_course_control',...
    'position',[pos(1)+5 pos(2)+5 120 25], ...
    'string',z, ...
    'fontsize',text_size,...
    'callback',callbackStr,...
    'value',2,...
		     'tooltipstring','Display time course window',...
		     'foregroundcolor',RGB_text,...
		     'backgroundcolor',RGB_background);

  pos = get(ih(FMRI.dime_z+1),'position');
  callbackStr = 'map_on_struc(''refresh'');';
  ed_lim = uicontrol('style','edit',...
    'unit','normalized',...
    'position',[pos(1)-.025 pos(2)+pos(4)+.06 .06 .05],...
    'tag','ed_lim',...
    'backgroundcolor',RGB_mainwin,...
		     'foregroundcolor',RGB_text,...
		     'tooltipstring','Maximum ROA plotting limit (|z|>0)',...
    'fontsize',text_size-2,...
    'callback',callbackStr,...
    'string','8.0');
  ed_thre = uicontrol('style','edit',...
    'unit','normalized',...
    'position',[pos(1)-.025 pos(2)+pos(4)+.01 .06 .05],...
    'tag','ed_thre',...
    'backgroundcolor',RGB_mainwin,...
		      'foregroundcolor',RGB_text,...
		      'tooltipstring','Minimum ROA limit (|z|>=0)',...
    'fontsize',text_size-2,...
    'callback',callbackStr,...
    'string','1.5');  

  display_component_map(figNumber,ih,u,component_displayed,invW);  
  % save the temporary data in 'userdata' of 'gcf'
  a = struct('figNumber', figNumber, ...
    'sourceData', u, ...
    'imageHandle', ih, ...
    'component_displayed', component_displayed, ...
    'total_component', total_component, ...
    'invW',invW,...
    'component_selected', component_selected,...
	     'component_rejected',component_rejected);
  set(figNumber, 'userdata', a);

  progressbar(1);
  pause(0.5);
  progressbar(nan);
  
  set(figNumber,'pointer','arrow');

elseif strcmp(action,'prev_component'),
  ff = gcf;
  set(ff,'pointer','watch');
  a = get(gcf,'userdata');
  figNumber = a.figNumber;
  ih = a.imageHandle;
  component_displayed = a.component_displayed;
  u = a.sourceData;
  invW = a.invW;
  total_component = a.total_component;
  component_selected = a.component_selected;
  component_rejected = a.component_rejected;
  component_displayed = component_displayed - 1;
  if component_displayed<1,
    component_displayed = total_component;
  end

  b = findobj('tag','ed_gotocomp');
  set(b,'string',int2str(component_displayed));

  display_component_map(figNumber,ih,u,component_displayed,invW);
  
  a = struct('figNumber', figNumber, ...
    'sourceData', u, ...
    'imageHandle', ih, ...
    'component_displayed', component_displayed, ...
    'total_component', total_component, ...
    'invW',invW,...
    'component_selected', component_selected,...
	     'component_rejected',component_rejected);
  set(figNumber, 'userdata', a);
  set(ff,'pointer','arrow');

elseif strcmp(action,'next_component'),
  ff = gcf;
  set(ff,'pointer','watch');
  a = get(gcf,'userdata');
  figNumber = a.figNumber;
  ih = a.imageHandle;
  component_displayed = a.component_displayed;
  u = a.sourceData;
  total_component = a.total_component;
  invW = a.invW;
  component_selected = a.component_selected;
  component_rejected = a.component_rejected;
  component_displayed = component_displayed + 1;
  if component_displayed>total_component,
    component_displayed = 1;
  end

  b = findobj('tag','ed_gotocomp');
  set(b,'string',int2str(component_displayed));
  
  display_component_map(figNumber,ih,u,component_displayed,invW); 

  a = struct('figNumber', figNumber, ...
    'sourceData', u, ...
    'imageHandle', ih, ...
    'component_displayed', component_displayed, ...
    'total_component', total_component, ...
    'invW',invW,...
    'component_selected', component_selected,...
	     'component_rejected',component_rejected);
  set(figNumber, 'userdata', a); 
  set(ff,'pointer','arrow');

elseif strcmp(action,'remove_list'),
  ff = gcf;
  set(ff,'pointer','watch');
  a = get(gcf,'userdata');
  figNumber = a.figNumber;
  ih = a.imageHandle;
  component_displayed = a.component_displayed;
  u = a.sourceData;
  total_component = a.total_component;
  component_selected = a.component_selected;
  component_rejected = a.component_rejected;
  invW = a.invW;

  if length(component_selected) == 0,
    %errordlg('No items in list');
    fmrlab_messagedlg('Component ROAs on Struc',...
		  'Error:',...
		  'No items in list');
    return;
  end

  a = findobj(figNumber,'tag','comp_select');
  b = str2num(get(a,'string'));
  c = get(a,'value');
  
  %find the selected item in the component list and remove it
  b(c) = [];
  component_selected = b;
  FMRI.icacomp = b;
  if length(component_selected) == 0,
    comp_displayed = 1;
    set(a,'string','0');
    set(a,'value',1);
  else
    if c <= length(component_selected),
      component_displayed = b(c);
    else
      component_displayed = b(length(component_selected));
      set(a,'value',length(component_selected));
    end
  end
  display_component_map(figNumber,ih,u,component_displayed,invW); 
  
  %refresh the list items
  if length(component_selected) == 0,
    set(a,'string','0');
  else
    set(a,'string',num2str(component_selected));
  end

  %refresh the temporary structure
  a = struct('figNumber', figNumber, ...
    'sourceData', u, ...
    'imageHandle', ih, ...
    'component_displayed', component_displayed, ...
    'total_component', total_component, ...
    'invW',invW,...
    'component_selected', component_selected,...
	     'component_rejected',component_rejected);
  set(figNumber, 'userdata', a); 
  set(ff,'pointer','arrow');

elseif strcmp(action,'goto_list'),
  ff = gcf;
  set(ff,'pointer','watch');
  a = get(gcf,'userdata');
  figNumber = a.figNumber;
  ih = a.imageHandle;
  component_displayed = a.component_displayed;
  u = a.sourceData;
  total_component = a.total_component;
  invW = a.invW;
  component_selected = a.component_selected;
  component_rejected = a.component_rejected;
  
  if length(component_selected) == 0,
    %errordlg('No items in list ...');
    fmrlab_messagedlg('Component ROAs on Struc',...
		  'Error:',...
		  'No items in list');
    return;
  end

  b = findobj(figNumber, 'tag', 'comp_select');
  aa = get(b,'string');
  component_displayed = component_selected(get(b,'value'));
  
  display_component_map(figNumber,ih,u,component_displayed,invW); 

  b = findobj(figNumber,'tag','ed_gotocomp');
  set(b,'string',int2str(component_displayed));
  
  a = struct('figNumber', figNumber, ...
    'sourceData', u, ...
    'imageHandle', ih, ...
    'component_displayed', component_displayed, ...
    'total_component', total_component, ...
    'invW',invW,...
    'component_selected', component_selected,...
	     'component_rejected',component_rejected);
  set(figNumber, 'userdata', a); 
  set(ff,'pointer','arrow');

elseif strcmp(action, 'goto_comp'),
  ff = gcf;
  set(ff,'pointer','watch');
  a = get(gcf,'userdata');
  figNumber = a.figNumber;
  ih = a.imageHandle;
  component_displayed = a.component_displayed;
  u = a.sourceData;
  total_component = a.total_component;
  invW = a.invW;
  component_selected = a.component_selected;
  component_rejected = a.component_rejected;
  
  b = findobj('tag','ed_gotocomp');
  component_displayed = str2num(get(b,'string'));
  if component_displayed > total_component | component_displayed < 1,
    %errordlg('Selected component is out of range');
    fmrlab_messagedlg('Component ROAs on Struc',...
		  'Error:',...
		  'Selected component is out of range');
    component_displayed = a.component_displayed;
    set(b,'string',num2str(component_displayed));
    return;
  end

  display_component_map(figNumber,ih,u,component_displayed,invW); 

  a = struct('figNumber', figNumber, ...
    'sourceData', u, ...
    'imageHandle', ih, ...
    'component_displayed', component_displayed, ...
    'total_component', total_component, ...
    'invW',invW,...
    'component_selected', component_selected,...
	     'component_rejected',component_rejected);
  set(figNumber, 'userdata', a); 
  set(ff,'pointer','arrow');
  
elseif strcmp(action,'refresh'),
  ff = gcf;
  set(ff,'pointer','watch');
  a = get(gcf,'userdata');
  figNumber = a.figNumber;
  ih = a.imageHandle;
  component_displayed = a.component_displayed;
  u = a.sourceData;
  invW = a.invW;
  total_component = a.total_component;
  component_selected = a.component_selected;
  component_rejected = a.component_rejected;

  display_component_map(figNumber,ih,u,component_displayed,invW); 
  set(ff,'pointer','arrow');

elseif strcmp(action,'add_list'),
  a = get(gcf,'userdata');
  figNumber = a.figNumber;
  ih = a.imageHandle;
  component_displayed = a.component_displayed;
  u = a.sourceData;
  total_component = a.total_component;
  invW = a.invW;
  component_selected = a.component_selected;
  component_rejected = a.component_rejected;
  
  b = findobj(figNumber,'tag','comp_select');
  aa = get(b,'string');
  aa = str2num(aa);
  found = 0;
  for i=1:length(aa),
    if component_displayed == aa(i),
      found = 1;
    end
  end
  
  if found == 1,
    str = ['Component ' num2str(component_displayed) ' already in list'];
    %errordlg(str);
    fmrlab_messagedlg('Component ROAs on Struc',...
		  'Error:',...
		  str);
    return;
  end

  if length(component_selected) == 0,
    set(b,'string',num2str(component_displayed));
  else
    aa = str2num(get(b,'string'));
    aa = num2str([aa; component_displayed]);
    set(b,'string',aa);
  end
  component_selected = [component_selected; component_displayed]; 
  
  a = struct('figNumber', figNumber, ...
    'sourceData', u, ...
    'imageHandle', ih, ...
    'component_displayed', component_displayed, ...
    'total_component', total_component, ...
    'invW',invW,...
    'component_selected', component_selected,...
	     'component_rejected',component_rejected);
  set(figNumber, 'userdata', a);

  FMRI.icacomp = component_selected; 

%----------------------------------
% adding component to rejected list
%----------------------------------
elseif strcmp(action,'add_rej_list'),
  a = get(gcf,'userdata');
  figNumber = a.figNumber;
  ih = a.imageHandle;
  component_displayed = a.component_displayed;
  u = a.sourceData;
  total_component = a.total_component;
  invW = a.invW;
  component_selected = a.component_selected; 
  component_rejected = a.component_rejected;
  
  b = findobj(figNumber,'tag','rej_list');
  aa = get(b,'string');
  aa = str2num(aa);
  found = 0;
  for i=1:length(aa),
    if component_displayed == aa(i),
      found = 1;
    end
  end
  if found == 1,
    str = ['Component ' num2str(component_displayed) ' already in list'];
    %errordlg(str);
    fmrlab_messagedlg('Component ROAs on Struc',...
		  'Error:',...
		  str);
    return;
  end

  if length(component_rejected) == 0,
    set(b,'string',num2str(component_displayed));
  else
    aa = str2num(get(b,'string'));
    aa = num2str([aa; component_displayed]);
    set(b,'string',aa);
  end
  component_rejected = [component_rejected; component_displayed]; 
  
  a = struct('figNumber', figNumber, ...
    'sourceData', u, ...
    'imageHandle', ih, ...
    'component_displayed', component_displayed, ...
    'total_component', total_component, ...
    'invW',invW,...
    'component_selected', component_selected,...
	     'component_rejected',component_rejected);
  set(figNumber, 'userdata', a);

  FMRI.rejcomp = component_rejected;
  
%------------------------------------------------
% display component selected in the rejected list
%------------------------------------------------
elseif strcmp(action,'goto_rej_list'),
  a = get(gcf,'userdata');
  figNumber = a.figNumber;
  ih = a.imageHandle;
  component_displayed = a.component_displayed;
  u = a.sourceData;
  total_component = a.total_component;
  invW = a.invW;
  component_selected = a.component_selected;
  component_rejected = a.component_rejected;
  
  if length(component_rejected) == 0,
    %errordlg('No items in list');
    fmrlab_messagedlg('Component ROAs on Struc',...
		  'Error:',...
		  'No items in list');
    return;
  end

  b = findobj(figNumber, 'tag', 'rej_list');
  aa = get(b,'string');
  component_displayed = component_rejected(get(b,'value'));
  
  display_component_map(figNumber,ih,u,component_displayed,invW); 

  b = findobj(figNumber,'tag','ed_gotocomp');
  set(b,'string',int2str(component_displayed));
  
  a = struct('figNumber', figNumber, ...
    'sourceData', u, ...
    'imageHandle', ih, ...
    'component_displayed', component_displayed, ...
    'total_component', total_component, ...
    'invW',invW,...
    'component_selected', component_selected,...
	     'component_rejected',component_rejected);
  set(figNumber, 'userdata', a); 
  
%----------------------------------------
% remove component from the rejected list
%----------------------------------------
elseif strcmp(action,'rm_rej_list'),
  a = get(gcf,'userdata');
  figNumber = a.figNumber;
  ih = a.imageHandle;
  component_displayed = a.component_displayed;
  u = a.sourceData;
  total_component = a.total_component;
  component_selected = a.component_selected;
  component_rejected = a.component_rejected;
  invW = a.invW;

  if length(component_rejected) == 0,
    %errordlg('No items in list');
    fmrlab_messagedlg('Component ROAs on Struc',...
		  'Error:',...
		  'No items in list');
    return;
  end

  a = findobj(figNumber,'tag','rej_list');
  b = str2num(get(a,'string'));
  c = get(a,'value');
  
  %find the selected item in the component list and remove it
  b(c) = [];
  component_rejected = b;
  FMRI.rejcomp = b;
  if length(component_rejected) == 0,
    comp_displayed = 1;
    set(a,'string','0');
    set(a,'value',1);
  else
    if c <= length(component_rejected),
      component_displayed = b(c);
    else
      component_displayed = b(length(component_rejected));
      set(a,'value',length(component_rejected));
    end
  end
  display_component_map(figNumber,ih,u,component_displayed,invW); 
  
  %refresh the list items
  if length(component_rejected) == 0,
    set(a,'string','0');
  else
    set(a,'string',num2str(component_rejected));
  end

  %refresh the temporary structure
  a = struct('figNumber', figNumber, ...
    'sourceData', u, ...
    'imageHandle', ih, ...
    'component_displayed', component_displayed, ...
    'total_component', total_component, ...
    'invW',invW,...
    'component_selected', component_selected,...
	     'component_rejected',component_rejected);
  set(figNumber, 'userdata', a); 

end

%================================================
function display_component_map(h0,ih,u,comp,invW)
%================================================
% display the component map on the functional images
% according to the local variable 'component_displayed'

global FMRI
global BASEIMG
global struc_image_width
global struc_image_height

RGB_mainwin = [1 .984 .895];
RGB_background = [0.860 0.824 0.618];
RGB_button = [0.860 0.824 0.618];
RGB_text = [0.333 0.283 0.003];

if isunix,
    text_size = 14;
else
    text_size = 12;
end

a = findobj(h0,'tag','ed_thre');
uthre = str2num(get(a,'string'));
lthre = -1 * uthre;
a = findobj(h0,'tag','ed_lim');
ulim = str2num(get(a,'string'));
llim = -1 * ulim;

% update ticks of colorbar
cb = linspace(ulim,uthre,5);
cb = [cb linspace(lthre,llim,5)];
cb = num2str(cb','%4.1f');
i = [2:4 7:9];
for j=1:size(cb,2),
  cb(i,j) = ' ';
end
set(ih(FMRI.dime_z+1),'yticklabel',cb);

temp_data = u(comp, :);
mean_temp_data = mean(temp_data);
std_temp_data = std(temp_data);
temp_data = (temp_data-mean_temp_data)/std_temp_data; %convert to z score

uidx = find(temp_data >= uthre);
temp_data1 = zeros(FMRI.dime_x*FMRI.dime_y*...
		   FMRI.dime_z,1);
for i=1:length(FMRI.idx),
  temp_data1(FMRI.idx(i)) = temp_data(i);
end
temp_data1 = reshape(temp_data1,FMRI.dime_x, FMRI.dime_y, ...
	       FMRI.dime_z);
if struc_image_width ~= FMRI.dime_x,
  xs = linspace(1, struc_image_width, FMRI.dime_x);
  ys = linspace(1, struc_image_height, FMRI.dime_y);    
  [xi, yi] = meshgrid(1:struc_image_width, 1:struc_image_height); 
  smap = [];
  for i=1:FMRI.dime_z,
    temp_data2 = interp2(xs, ys, temp_data1(:,:,i), xi, yi);
    temp_data2 = reshape(temp_data2, struc_image_width*struc_image_height, 1);
    smap = [smap temp_data2];
  end
else
  smap = temp_data1;
end
smap = reshape(smap,struc_image_width*struc_image_height*...
	       FMRI.dime_z,1);
pos_smap = smap;
ii = find(pos_smap > ulim);
pos_smap(ii) = ulim;
ii = find(pos_smap < uthre);
pos_smap(ii) = 0.0;
ii = find(pos_smap >= uthre);
pos_smap(ii) = fmrlab_scale(pos_smap(ii),99,128);
BASEIMG = reshape(BASEIMG,struc_image_width*struc_image_height*FMRI.dime_z,1);
img = max(BASEIMG,pos_smap);

neg_smap = smap;
ii = find(neg_smap < llim);
neg_smap(ii) = llim;
ii = find(neg_smap > lthre);
neg_smap(ii) = 0.0;
ii = find(neg_smap <= lthre);
neg_smap(ii) = fmrlab_scale(neg_smap(ii),65,94);
img = max(img,neg_smap);

img = reshape(img,struc_image_width,struc_image_height,FMRI.dime_z);
 
for i=1:FMRI.dime_z,
  axes(ih(i));
  image(img(:,:,i)');
  axis off
  axis image
end
fmrlab_color(2);

figure(h0);
set(ih(FMRI.dime_z+2),'string',['Component #' num2str(comp)]);

pos = get(h0,'position');
h_obj = findobj(h0,'tag','time_course_control');
disp_tc = get(h_obj,'value');
if disp_tc == 1,
  figTimeCourse = findobj('name','Component Time Course');
  if figTimeCourse ~= 0,
    close(figTimeCourse);
  end
  figBOLDImage = findobj('name','BOLD Image');
  if figBOLDImage ~= 0,
    close(figBOLDImage);
  end
else
  figTimeCourse = findobj('name','Component Time Course');
  if isempty(figTimeCourse),
    figTimeCourse = figure(...
      'name', 'Component Time Course', ...
      'numbertitle','off',...
      'doublebuffer', 'on', ...
      'color',RGB_mainwin,...
	'toolbar','none',...
      'position', [pos(1) pos(2)-275 pos(3) 250],...
      'backingstore','off',...
	'menubar','none');
  else
    figure(figTimeCourse);
    clf
  end
  
  switch(disp_tc),
    case 2,
      figBOLDImage = findobj('name','BOLD Image');
      if figBOLDImage ~= 0,
        close(figBOLDImage);
      end
      ddd = invW(:,comp) * u(comp,:);
      ddd = ddd(:,uidx)';
      tc = mean(ddd);
      ddd2 = FMRI.data(uidx);
      tc2 = mean(ddd2');
      tc = 100 * (tc-mean(tc)) / mean(tc2);
      if isempty(FMRI.tr) | FMRI.tr == 0,
        plot(tc);
        set(gca,'fontsize',text_size);
        xlabel('Scan Number');
      else
        x1 = linspace(FMRI.tr,FMRI.timepoint*FMRI.tr,FMRI.timepoint);
        plot(x1,tc);
        set(gca,'fontsize',text_size);
        xlabel('Time (s)');
      end
      legend('Comp. proj. to +ROA');
      ylabel('\Delta BOLD Signal Change (%)');
      pos = get(gca,'position');
      pos(2) = pos(2) + pos(4)*0.1;
      pos(4) = pos(4)*0.9;
      set(gca,'position',pos);

     case 3,
      figBOLDImage = findobj('name','BOLD Image');
      if figBOLDImage ~= 0,
        close(figBOLDImage);
      end
      ddd = FMRI.data(:,uidx)';
      tc = mean(ddd);
      tc1 = tc(1,:);
      tc1 = 100 * (tc1-mean(tc1)) / mean(tc1);
      if isempty(FMRI.tr) | FMRI.tr == 0,
        plot(tc1,'r');
        set(gca,'fontsize',text_size);
        xlabel('Scan Number');
      else
        x1 = linspace(FMRI.tr,FMRI.timepoint*FMRI.tr,FMRI.timepoint);
        plot(x1,tc1,'r');
        set(gca,'fontsize',text_size);
        xlabel('Time (s)');
      end
      legend('+ROA-mean time course');
      ylabel('\Delta BOLD Signal Change (%)');
      pos = get(gca,'position');
      pos(2) = pos(2) + pos(4)*0.1;
      pos(4) = pos(4)*0.9;
      set(gca,'position',pos);
      
    case 4,
      figBOLDImage = findobj('name','BOLD Image');
      if figBOLDImage ~= 0,
        close(figBOLDImage);
      end
      % calculating the back projection of specified ICA component
      ddd = invW(:,comp) * u(comp,:);
      ddd = ddd(:,uidx)';
      
      tc = mean(ddd);
      % tc is the time course obtained from back-projection

      ddd2 = FMRI.data(:,uidx);
      
      tc2 = mean(ddd2');
      % tc2 is the time course obtained from the raw data

      grand_mean = mean(tc2);
      tc = 100 * (tc-mean(tc)) / grand_mean;
      if isempty(FMRI.tr) | FMRI.tr == 0,
        plot(tc);
        set(gca,'fontsize',text_size);
        xlabel('Scan Number');
      else
        x1 = linspace(FMRI.tr,FMRI.timepoint*FMRI.tr,FMRI.timepoint);
        plot(x1,tc);
        set(gca,'fontsize',text_size);
        xlabel('Time (s)');
      end
      ylabel('\Delta BOLD Signal Change (%)');

      % calculating the ROA mean time course
%      ax1 = gca;
%      set(ax1,'xcolor','b','ycolor','b','fontsize',text_size);
%      ax2 = axes('position',get(ax1,'position'),...
%        'yaxislocation','right',...
%        'color','none','ycolor','r',...
%        'fontsize',text_size);
%      ylabel('Signal Change (%)');
      ddd = FMRI.data(:,uidx)';
      tc2 = mean(ddd);
      tc2 = 100 * (tc2-mean(tc2)) / grand_mean;
      x2 = linspace(FMRI.tr,FMRI.timepoint*FMRI.tr,FMRI.timepoint);
      hold on
%      line(x2,tc2,'parent',ax2,'color','r');
      plot(x2,tc2,'r');
      legend('Comp. proj. to +ROA','+ROA-mean time course');
      pos = get(gca,'position');
      pos(2) = pos(2) + pos(4)*0.1;
      pos(4) = pos(4)*0.9;
      set(gca,'position',pos);
      
      v = 100*((1-var(tc2-tc)/var(tc2)));
      % (raw - back_projection) / raw

      ti = text(0.75*max(x2),0.95*min(str2num(get(gca,'yticklabel'))),[sprintf('p.v.a.f. = %5.2f',v) '%']);
      set(ti,'fontsize',text_size,'color',[1 0 1]);

    case 5,
      % plot the component time course,
      ddd = invW(:,comp) * u(comp,:);
      ddd = ddd(:,uidx)';
      tc = mean(ddd);
      ddd2 = FMRI.data(uidx);
      tc2 = mean(ddd2');
      tc = 100 * (tc-mean(tc)) / mean(tc2);
      if isempty(FMRI.tr) | FMRI.tr == 0,
        plot(tc);
        set(gca,'fontsize',text_size);
        xlabel('Scan Number');
      else
        x1 = linspace(FMRI.tr,FMRI.timepoint*FMRI.tr,FMRI.timepoint);
        plot(x1,tc);
        set(gca,'fontsize',text_size);
        xlabel('Time (s)');
      end
      ylabel('\Delta BOLD Signal Change (%)');
      pos = get(gca,'position');
      pos(2) = pos(2) + pos(4)*0.1;
      pos(4) = pos(4)*0.9;
      set(gca,'position',pos);
      legend('Comp. proj. to +ROA');
      
      % create another window to plot BOLD image
      figBOLDImage = findobj('name','BOLD Image');
      BOLD_err = 0;
      if figBOLDImage ~= 0,
        figure(figBOLDImage);
      else
        pos = get(figTimeCourse,'position');
        figBOLDImage = figure(...
          'name', 'BOLD Image', ...
          'numbertitle','off',...
          'doublebuffer', 'on', ...
          'color',RGB_mainwin,...
          'position', [pos(1)+pos(3)+10 pos(2) 440 500],...
          'backingstore','off',...
	    'toolbar','none',...
	    'menubar','none');     
        % input necessary parameters for BOLD image plot
        prompt = {'Trial onsets (scan numbers):',...
		  'Trial scans (in scans):'};
        defs = {num2str(FMRI.soa), num2str(FMRI.trial_length)};
        a = fmrlab_inputdlg(prompt,'Bold-image plot parameters',1,defs); 
 	if ~isempty(a),
          FMRI.soa = str2num(char(a(1)));
	  if length(FMRI.soa)==1 | length(FMRI.soa)==0,
	    %errordlg('Too few trials');
	    fmrlab_messagedlg('Component ROAs on Struc',...
			  'Error:',...
			  'Too few trials');
	    BOLD_err = 1;
	  end
	  
	  for i=1:length(FMRI.soa),
	    if FMRI.soa(i) > FMRI.timepoint | FMRI.soa(i) < 1,
	      %errordlg('Onset scan number exceeds total scan
              %number');
	      fmrlab_messagedlg('Component ROAs on Struc',...
			    'Error:',...
			    'Onset scan number exceeds total scan number');
	      BOLD_err = 1;
	      break;
	    end
	  end
	  
          FMRI.trial_length = str2num(char(a(2)));
	  if FMRI.trial_length == 0,
	    %errordlg('Trial length cannot be ZERO');
	    fmrlab_messagedlg('Component ROAs on Struc',...
			  'Error:',...
			  'Trial length cannot be zero');
	    BOLD_err = 1;
	  end
	  
	  for i=1:length(FMRI.soa),
	    temp = FMRI.trial_length + FMRI.soa(i) - 1;
	    if temp > FMRI.timepoint,
	      %errordlg(['Trial length exceeds the total scan' ...
		%	' number']);
	      fmrlab_messagedlg('Component ROAs on Struc',...
			    'Error:',...
			    'Trial length exceeds total scan number');
	      BOLD_err = 1;
	      break;
	    end
	  end
	else      
	  %errordlg('Stopped by user');
	  BOLD_err = 1;
        end
      end
      if BOLD_err == 1,
	h_obj = findobj(h0,'tag','time_course_control');
	set(h_obj,'value',1);
	close(figTimeCourse);
	close(figBOLDImage);
	return;
      end
      
      % BOLD image plot
      a = [];
      for i=1:length(FMRI.soa),
        a = [a tc(FMRI.soa(i):FMRI.soa(i)+FMRI.trial_length-1)];
      end
      figure(figBOLDImage);
      clf
      boldimage(a,zeros(length(FMRI.soa),1),...
                0:FMRI.tr:FMRI.tr*(length(FMRI.trial_length)-1),...
                ['Component #' num2str(comp)],1,1,'erp','nosort','noplot','cbar');
  end
  figure(figTimeCourse);
  ti = textsc(['Component #' num2str(comp)],'title');
  set(ti,'fontsize',text_size+2);
end
