% pvaf_vs_thre() - calculate pvaf according to the specified 
%                  threshold of z value and make a pvaf vs
%                  thre plot.
%
% Usage: >> pvaf_vs_thre;
%
% Authors: Jeng-Ren Duann, CNL/Salk Inst. 2003-01-20
%
% See also: pvafmap_ui(), map_on_fmri()

% Copyright (C) 2002 Jeng-Ren Duann, Salk Institute, duann@salk.edu
%
% This program is free software; you can redistribute it and/or
% modify it under the terms of the GNU General Public License as
% published by the Free Software Foundation; either version 2 of
% the License, or (at your option) any later version.
%
% This program is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU General Public License for more details.
%
% You should have received a copy of the GNU General Public License
% along with this program; if not, write to the Free Software
% Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

% $Log: pvaf_vs_thre.m,v $
% Revision 1.1  2003/02/06 19:13:25  duann
% Initial revision
%

function pvaf_vs_thre()
  
  global FMRI

  RGB_mainwin = [1 .984 .895];
  RGB_background = [0.860 0.824 0.618];
  RGB_button = [0.860 0.824 0.618];
  RGB_text = [0.333 0.283 0.003];
  
  if isunix,
      text_size = 14;
  else
      text_size = 12;
  end

  % look up map_on_fmri window for component number, which is
  % diaplyed on the screen now.
  h = findobj('name', 'Display Component ROA on Functional Variance Images');
  if h==0,
    fmrlab_messagedlg('PVAF versus Z Threshold',...
		  'Error:',...
		  'Window not found');
    return;
  end
  hh = findobj(h, 'tag','ed_component');
  if hh == 0,
    fmrlab_messagedlg('PVAF versus Z Threshold',...
		  'Error:',...
		  'Component number not found');
    return;
  end
  comp_no = str2num(get(hh,'string'));

  set(gcf,'pointer','watch');
  
  W = FMRI.icaweight * FMRI.icasphere;
  invW = pinv(W);
  u = W * FMRI.data;
  temp = u(comp_no,:);
  temp = (temp-mean(temp))/std(temp); %convert to z value
  thre = 0.5:0.1:6;
  v = [];
  n_plus = [];
  n_minus = [];
  for i=1:length(thre),
    uidx = find(temp > thre(i));
    n_plus = [n_plus length(uidx)];
    lidx = find(temp < (-1.0*thre(i)));
    n_minus = [n_minus length(lidx)];
    ddd = invW(:,comp_no) * u(comp_no,:);
    ddd = ddd(:,uidx);
    bpctc = mean(ddd');
    ddd = FMRI.data(:,uidx);
    roamtc = mean(ddd');
    grand_mean = mean(roamtc);
    roamtc = 100 * (roamtc-grand_mean)/grand_mean;
    bpctc = 100 * (bpctc - mean(bpctc))/grand_mean;
    a = var(roamtc);
    v = [v 100*(1-var(roamtc-bpctc)/a)];
  end
  
  v2 = [];
  for i=0.5:0.1:6
    uidx = find(temp >= i-0.25 & temp < i+0.25);
    ddd = invW(:,comp_no) * u(comp_no,:);
    ddd = ddd(:,uidx);
    bpctc = mean(ddd');
    ddd = FMRI.data(:,uidx);
    roamtc = mean(ddd');
    grand_mean = mean(roamtc);
    roamtc = 100 * (roamtc-grand_mean)/grand_mean;
    bpctc = 100 * (bpctc - mean(bpctc))/grand_mean;
    a = var(roamtc);
    v2 = [v2 100*(1-var(roamtc-bpctc)/a)];
  end

  set(gcf,'pointer','arrow');
  
  f1 = figure('name', 'PVAF Histogram',...
	      'numbertitle', 'off',...
	      'color',RGB_mainwin,...
	      'toolbar','none',...
	      'menubar','none');
  pos = get(f1,'position');
  pos(1) = 100;
  pos(2) = 50;
  pos(3) = pos(3)*.75;
  pos(4) = pos(4)*1.5;
  set(f1,'position',pos);
  subplot(2,1,1);
  ll = plot(thre, v);
  set(ll,'linewidth',2);
  set(gca,'fontsize',text_size-2,...
	  'linewidth',2);
  
  ti = '% ROA-mean variance accounted for (PVAF)';
  ti = title(ti);
  set(ti, 'fontsize', 14);
  set(get(gca,'xlabel'),...
      'fontsize',text_size,...
      'string','ROA Threshold (z)');
  set(get(gca,'ylabel'),...
      'fontsize',text_size,...
      'string','ROA PVAF (%)');
  [maxv,ii] = max(v);
  hold on
  v2_tt = 0.5:0.1:6;
  ll = plot(v2_tt,v2,'r');
  set(ll,'linewidth',2);
  legend('Thresh.->Max. ROA', 'moving-window ROA');
  plot(thre(ii),maxv,'rx');
  str = sprintf('(%5.2f,%5.2f)',thre(ii),maxv);
  ti=text(thre(ii)+0.2,maxv,str);
  set(ti,'color',[1 0 0]);
 
  hh = findobj('name',...
	       'Display Component ROA on Functional Variance Images');
  if ~isempty(hh)
    hh1 = findobj(hh,'tag','ed_thre');
    temp = str2num(get(hh1,'string'));
    for i=1:length(thre),
      if thre(i) == temp,
	thre_idx = i;
	break;
      end
    end
  end
  ax = axis;
  ll = line([temp temp],[ax(3) ax(4)]);
  set(ll,'linestyle','--',...
	 'color','mag',...
	 'linewidth',2);
  plot(temp,v(thre_idx),'mo');
  str = sprintf(' %5.2f%%',v(thre_idx));
  tx = text(temp,v(thre_idx),str);
  set(tx,'fontsize',text_size-2);
  plot(temp,v2(thre_idx),'mo');
  str = sprintf(' %5.2f%%',v2(thre_idx));
  tx = text(temp,v2(thre_idx),str);
  set(tx,'fontsize',text_size-2);
  
  subplot(2,1,2);
  ll =  plot(thre,n_plus,'r');
  set(ll,'linewidth',2);
  hold on
  ll = plot(thre,n_minus);
  set(ll,'linewidth',2);
  set(gca,'fontsize',text_size-2,...
	  'linewidth',2);
  set(get(gca,'xlabel'),...
      'fontsize',text_size,...
      'string','ROA Threshold (z)');
  set(get(gca,'ylabel'),...
      'fontsize',text_size,...
      'string','Number of Voxels');
  ax = axis;
  ll = line([temp temp],[ax(3) ax(4)]);
  set(ll,'linestyle','--',...
	 'color','mag',...
	 'linewidth',2);
  plot(temp,n_plus(thre_idx),'mo');
  str = sprintf(' %5.2f',n_plus(thre_idx));
  tx = text(temp,n_plus(thre_idx),str);
  set(tx,'fontsize',text_size-2);
  plot(temp,n_minus(thre_idx),'mo');
  str = sprintf(' %5.2f',n_minus(thre_idx));
  tx = text(temp,n_minus(thre_idx),str);
  set(tx,'fontsize',text_size-2);
  legend('+ROA','-ROA');
  
  ti = title('Number of ROA voxels at different thresholds');
  set(ti,'fontsize',text_size);
  
  ti = sprintf('Component (%d)',comp_no);
  ti = textsc(0.5,0.98,ti);
  set(ti,'fontsize',text_size+2,'fontweight','bold');