% pvafmap_ui() - calculate PVAF for each voxel per every component,
%                assign each voxel a component which has the
%                maximal PVAF for that voxel, and popup two figures
%                to show the values of maximal PVAF image and
%                component image
% 
% Usage: 
%   >> pvafmap_ui(train_data, train_idx, wgt, sph, dime, tm,
%                 comp_opt);
%
% Inputs:
%   train_data  - 2D fMRI data within the brain area used by ICA
%                  training (time points, voxel number)
%    train_idx   - compressed coordinate (1D) of the voxels within the brain area
%    wgt         - weight matrix obtained by ICA training
%    sph         - sphere matrix obtained by ICA training
%    dime        - matrix contains the information for image dimension, should be 
%                  [image_width image_height image_depth]
%    tm          - tick label for time interval instead of time point
%    comp_opt    - component option 
%                  1: all the components; 
%                  2: excluding the rejected components; (FMRI.rejcomp) 
%                  3: using only the selected components
%                  (FMRI.icacomp)
%
% Outputs:
%
% Author: Jeng-Ren Duann, CNL/Salk Inst. & SCCN/INC/UCSD, 2002
%
% See also: roamap_ui(), roaproj_ui()

% --
% Copyright (C) 2002 Jeng-Ren Duann, Salk Institute, duann@salk.edu
%
% This program is free software; you can redistribute it and/or
% modify it under the terms of the GNU General Public License as
% published by the Free Software Foundation; either version 2 of
% the License, or (at your option) any later version.
%
% This program is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU General Public License for more details.
%
% You should have received a copy of the GNU General Public License
% along with this program; if not, write to the Free Software
% Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

% $Log: pvafmap_ui.m,v $
% Revision 1.1  2003/02/06 19:13:18  duann
% Initial revision
%

function pvafmap_ui(train_data,train_idx,wgt,sph,dime,tm,comp_opt)

% Jeng-Ren Duann, CNL/Salk Inst. 2002-05-01
%    add input argument 'comp_opt', J.-R. 2002-05-20
%    change window color, J.-R. 2002-10-11
  
  global FMRI
  
  if length(FMRI) == 0,
    fmrlab_messagedlg('Component ROA (PVAF)',...
	       'Error:',...
	       ['No current data set exists. Please load a ''.fmr''' ...
		' data files first']);
    return;
end

  RGB_mainwin = [1 .984 .895];
  RGB_background = [0.860 0.824 0.618];
  RGB_button = [0.860 0.824 0.618];
  RGB_text = [0.333 0.283 0.003];
  
  if isunix,
      text_size = 14;
  else
      text_size = 12;
  end

  if nargin < 1,
    train_data = FMRI.data;
  end
  if nargin < 2,
    train_idx = FMRI.idx;
  end
  if nargin < 3,
    wgt = FMRI.icaweight;
  end
  if nargin < 4,
    sph = FMRI.icasphere;
  end
  if nargin < 5,
    dime = [FMRI.dime_x FMRI.dime_y FMRI.dime_z];  
  end
  if nargin < 6,
    tm = linspace(0,FMRI.tr*(FMRI.timepoint-1),FMRI.timepoint);
  end
  
  if nargin < 7,
    prompt = sprintf(['Component to Include:\n  1: all\n  2: exclude' ...
		    [' rejected components\n  3: selected components' ...
		     ' only']]);
    a = fmrlab_inputdlg(prompt,'ROAMAP Display',1,{'1'});
    if isempty(a),
      %errordlg('Stopped by user!');
      return;
    end
    comp_opt = str2num(char(a));
  end

  % plot z score map for all ica components
  W = wgt*sph;
  u = W*train_data;
  invW = pinv(W);
  
  progressbar(0.5/size(W,1));
  
  sz = size(u);
  switch comp_opt,
   case 1
    disp('Display all components');
    comp_to_show = 1:sz(1);
   case 2
    disp('Exclude rejected components');
    comp_to_show = 1:sz(1);
    comp_to_show(FMRI.rejcomp) = [];
   case 3
    disp('Display selected components only');
    comp_to_show = FMRI.icacomp;
  end
  sz(1) = length(comp_to_show);
  
  vmap = -100*ones(1,sz(2));
  cmap = 101*ones(1,sz(2));
  for j=1:sz(1),
    u1 = zeros(size(u));
    u1(comp_to_show(j),:) = u(comp_to_show(j),:);
    x = invW * u1;
    for i=1:size(x,2),
      t1 = x(:,i);
      t2 = FMRI.data(:,i);
      vv = 100 * (1-var(t2-t1)/var(t2));
      if vv > vmap(i),
	vmap(i) = vv;
	cmap(i) = comp_to_show(j);
      end
    end
    progressbar(j/sz(1));
  end
  progressbar(nan);
  
  vvmap = zeros(1,FMRI.dime_x*FMRI.dime_y*FMRI.dime_z);
  ccmap = 101*ones(1,FMRI.dime_x*FMRI.dime_y*FMRI.dime_z);
  for i=1:length(FMRI.idx),
    vvmap(FMRI.idx(i)) = vmap(i);
    ccmap(FMRI.idx(i)) = cmap(i);
  end
  vvmap = reshape(vvmap,FMRI.dime_x,FMRI.dime_y,FMRI.dime_z);
  ccmap = reshape(ccmap,FMRI.dime_x,FMRI.dime_y,FMRI.dime_z);
  f_zmap = figure(...
      'name','Tiling Brain by ICA Components: Maximal p.v.a.f. Map',...
      'color',RGB_mainwin,...
      'numbertitle','off',...
      'toolbar','none',...
      'menubar','none');
  set(f_zmap,'numbertitle','off');
  disp_col = ceil(sqrt(dime(3)));
  disp_row = ceil(dime(3)/disp_col);
  for i=1:dime(3)
    subplot(disp_row,disp_col,i);
    image((vvmap(:,:,i)'));
    axis image; axis off
  end
  caxis([-100 100]);
  cbar;
  ti = textsc(0.5,0.95,'Largest pvaf Map');
  set(ti,'fontsize',text_size);

  f_comp = figure(...
      'name','Independent Component Regions of Activity',...
      'color',RGB_mainwin,...
      'toolbar','none',...
      'numbertitle','off',...
      'menubar','none');
  set(f_comp,'numbertitle','off');
  for i=1:dime(3)
    subplot(disp_row,disp_col,i);
    imagesc(ccmap(:,:,i)');
    axis image; axis off
  end
  colorm = colormap;
  colormap(flipud(colorm));
  cb = cbar;
  caxis([0 101]);
  pos = get(cb,'position');
  set(cb,'position',[pos(1)+pos(3) pos(2) pos(3) pos(4)]);
  ti = textsc(pos(1),pos(2)+pos(4)/2,'Component Number');
  set(ti,'rotation',90);
  ti = textsc(0.5,0.95,'Component with Largest pvaf at Each Voxel');
  set(ti,'fontsize',text_size);
  ti = textsc(0.5,0.08,...
	      'Click left button on slice to magnify and choose');
  set(ti,'verticalalignment','top');
  ti = textsc(0.5,0.04,...
	      'component of interest. Click right button to quit');
  set(ti,'verticalalignment','top');

  figure(f_comp);
  while 1
    [my,mx,mb] = ginput(1);
    if mb == 3,
      break;
    end
    if mb == 1,
      a = get(gco,'cdata');
      tempf = figure(...
	  'name','Select Component of Interest',...
	  'color',RGB_mainwin,...
	  'toolbar','none',...
	  'numbertitle','off',...
	  'menubar','none');
      imagesc(a);
      axis image
      axis off
      colorm = colormap;
      colormap(flipud(colorm));
      ti = title('Click left button to specify component of interest');
      cbar;
      caxis([0 101]);
%      [my,mx,mb] = ginput(1);
%      comp = a(round(mx),round(my));
      [mx,my,comp,mb]=mousemoving(tempf);
      close(tempf);
      if mb == 3,
        break;
      end
      if mb == 1,
        if (comp<1)
          disp(['Area with no value is specified, try again...']);
        else
	  z_thre = 1.5;
          disp(['Component: ' int2str(comp)]);
          tm = linspace(0,FMRI.tr*(FMRI.timepoint-1),FMRI.timepoint);
          v = roatc_ui(train_data,train_idx,wgt,sph,dime,comp,z_thre,3,tm);
	  b = zeros(size(a));
          ii = find(a<101 & a~=comp);
          b(ii) = 20;
	  ii = find(a==comp);
	  b(ii) = 80;
          sbplot(3,3,1);
          imagesc(b); axis image; axis off
          caxis([0 101]);

          % doing the roaproj stuff
          figure('name','Component ROA-Mean Time Course: Largest Component Contribution',...
		 'numbertitle','off',...
		 'toolbar','none',...
		 'color',RGB_mainwin,...
		 'menubar','none');
          sbplot(4,4,[6 11]);
          imagesc(b); axis off; axis image
	  caxis([0 101]);
          ax = get(gca,'position');
	  set(gca,'position',[ax(1) ax(2)+.12 ax(3) ax(4)]);
          t1 = title(['IC  ' int2str(comp)]);
          set(t1,'fontsize',text_size+2);
          ax = axes('position',[ax(1)+0.1*ax(3) ax(2)-.22 0.8*ax(3) .22]);
	  
          mm = u(comp,:);
          mm = (mm-mean(mm))/std(mm);

          ii = find(mm>=z_thre);
          c = (mean(train_data(:,ii)'));
          mean_c = mean(c);
          c = 100*(c-mean_c)/mean_c;
          plot(tm,c);
          set(gca, 'yticklabel',[]);
          [p,q] = sort(v);
          pos = [1 3 4 6];
          j = 1;
          r = c;
          ss = zeros(1,length(c));
          for i=length(q):-1:length(q)-3,
            u2 = zeros(size(u));
            u2(q(i),:) = u(q(i),:);
            data2 = invW(:,q(i))*u2(q(i),:); 
            tc2 = data2(:,ii)';
            mean_tc2 = (mean(tc2));
            mean_tc2 = 100*(mean_tc2 - mean(mean_tc2)) / mean_c;
            sbplot(2,3,pos(j));
            plot(tm,c);
            hold on
            plot(tm,mean_tc2,'r','linewidth',2);
            if (j==2 | j==4),
              set(gca,'yticklabel',[]);
            end
            if (j==1 | j==3),
              tx = get(gca,'ylabel');
              set(tx,'string','\Delta BOLD Signal (%)');
              set(tx, 'fontsize',text_size);
            end
            if (j==3 | j==4),
              tx = get(gca,'xlabel');
              set(tx,'string','Time (s)');
              set(tx, 'fontsize',text_size);
            end
            ti = sprintf('IC%d      %4.1f%%',q(i),p(i));
            tx = title(ti);
            set(tx,'fontsize',text_size);
            r = r-mean_tc2;
            ss = ss + mean_tc2;
            j = j+1;

    	    if comp == q(i),
	      set(get(gca,'title'),'fontweight','bold');
%              set(gca,'xcolor',[1 0 0]);
%              set(gca,'ycolor',[1 0 0]);
%              ca = get(gca,'ylabel');
%              set(ca,'color',[0 0 0]);
            end
          end
          axes(ax);
          hold on
          plot(tm,r,'k');
          plot(tm,ss,'r');
          ti = get(gca,'xlabel');
          set(ti,'string','Time (s)');
          set(ti, 'fontsize',text_size);

          vv = 100* (1-var(c-ss)/var(c));
          axes(ax);
          ax2 = axis;
          ti = sprintf('%4.1f%%',vv);
          tx = title(['All four combined  ' ti]);
          set(tx,'fontsize',text_size);
        end
      end

      figure(f_comp);
    end
  end


  