function [res]=est_mvarResiduals(X,AR,mu)
% 
%  Computes the M-variate residuals for variables k=1...M
% 
%    res(:,k) = X(:,k+p) - mu - A1*X(:,k+p-1) - ... - Ap*X(:,k)
%
%  of an VAR(p) model with AR=[A1 ... Ap]
%   
%   Inputs:
%       
%       X:      [M x npoints x ntrials] matrix of sensor data
%       AR:     VAR[p] model coefficients in format AR=[A1 ... Ap]
%       mu:     [1 x nchannels] vector of model intercepts (if no intercepts
%               estimated, set mu = zeros(1,nchannels) and process mean is assumed to 
%               be zero
%   Output:
% 
%       res:    [M x npoints-p x ntrials] matrix of residuals
%
% See Also: pop_est_fitMVAR()
%
% References:
% 
% [1] Mullen T (2010) The Source Information Flow Toolbox (SIFT):
%     Theoretical Handbook and User Manual. Chapter 4
%     Available at: http://www.sccn.ucsd.edu/wiki/Sift
% 
% Author: Tim Mullen 2010, SCCN/INC, UCSD. 
% Email:  tim@sccn.ucsd.edu

% This function is part of the Source Information Flow Toolbox (SIFT)
%
% This program is free software; you can redistribute it and/or modify
% it under the terms of the GNU General Public License as published by
% the Free Software Foundation; either version 3 of the License, or
% (at your option) any later version.
%
% This program is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU General Public License for more details.
%
% You should have received a copy of the GNU General Public License
% along with this program; if not, write to the Free Software
% Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

  nchs  = size(X,1);                    % number of observations
  n     = size(X,2);                    % number of variables
  ntr   = size(X,3);                    % number of realizations (trials)
  p     = size(AR,2)/nchs;              % order of model
  nres  = n-p;                          % number of residuals
  

  mu     = mu(:);                       % force mu to be column vector
  mu     = mu*ones(1,nres);             % construct matrix
  
  % Get time series of residuals 
  l = 1:nres;                           % vectorized loop l=1,...,nres 
  res = zeros(nchs,nres,ntr);
  for tr=1:ntr
      res(:,l,tr) = X(:,l+p,tr) - mu;
      for k=1:p
        res(:,l,tr) = res(:,l,tr) - AR(:, (k-1)*nchs+1:k*nchs)*X(:,l-k+p,tr);
      end
  end
  
%   % Center residuals by subtraction of the mean 
%   res   = res - repmat(mean(res,2),[1 nres 1]);