function res = hlp_aggregatestructs(structs,defaultop,varargin)
% Aggregate structs (recursively), using the given combiner operations.
% Result = hlp_aggregatestructs(Structs,Default-Op,Field-Ops...)
%
% This results ideally in a single struct which has aggregated values in its fields (e.g., arrays, averages, etc.).
%
% In:
%   Structs    : cell array of structs to be aggregated (recursively) into a single struct
%
%   Default-Op : optional default combiner operation to execute for every field that is not itself a struct;
%                see notes for the format;
%
%   Field-Ops  : name-value pairs of field-specific ops; names can have dots to denote operations that apply to subfields.
%                field-specific ops that apply to fields that are themselves structures become the default op for that sub-structure
%
% Out:
%   recursively merged structure.
%
% Notes:
%   If an operation cannot be applied, a sequence of fall-backs is silently applied. First, concatenation is tried, then,
%   replacement is tried (which never fails). Therefore, function_handles are being concatenated up to 2008a, and replaced starting with 2008b.
%   Operations are specified in one of the following formats:
%   * 'cat': concatenate values horizontally using []
%   * 'replace': replace values by those of later structs (noncommutative)
%   * 'sum': sum up values
%   * 'mean': compute the mean value
%   * 'std': compute the standard deviation
%   * 'median': compute the median value
%   * 'random': pick a random value
%   * 'fillblanks': replace [] by values of later structs
%   * binary function: apply the function to aggregate pairs of values; applied in this order f(f(f(first,second),third),fourth)...
%   * cell array of binary and unary function: apply the binary function to aggregate pairs of values, then apply the unary function to
%     finalize the result: functions {b,u} are applied in the following order: u(b(b(b(first,second),third),fourth))
%
%                                        Christian Kothe, Swartz Center for Computational Neuroscience, UCSD
%                                        2010-05-04

warning off MATLAB:warn_r14_function_handle_transition

if ~exist('defaultop','var')
    defaultop = 'cat'; end
if ~iscell(structs)
    structs = {structs}; end
fieldops = hlp_varargin2struct(varargin);

% translate all ops (if they are specified as strings)
defaultop = translateop(defaultop);
fieldops = translateop(fieldops);

% aggregate & finalize
res = finalize(aggregate(structs,defaultop,fieldops),defaultop,fieldops);

function res = aggregate(structs,defaultop,fieldops)
for k=find(cellfun(@length,structs)>1)
    % merge struct arrays in the inputs recursively
    structs{k} = hlp_aggregatestructs(structarray2cellarray(structs{k}),defaultop,fieldops); end
res = structs{1};
for i=2:length(structs)
    si = structs{i};
    for fn=fieldnames(si)'
        f = fn{1};
        if isfield(fieldops,f)
            % a field-specific op applies
            if isstruct(fieldops.(f))
                % ... which is itself a struct
                fop = fieldops.(f);
            else
                op = fieldops.(f);
            end
        else
            % the default op applies
            op = defaultop;
            fop = fieldops;
        end

        if isfield(res,f)
            % we have to combine
            if isstruct(res.(f)) && isstruct(si.(f))
                % recurse
                res.(f) = aggregate({res.(f),si.(f)},op,fop);
            else
                % aggregate
                try
                    res.(f) = op{1}(res.(f),si.(f));
                catch
                    % fallback...
                    try
                        res.(f) = [res.(f),si.(f)];
                    catch
                        res.(f) = si.(f);
                    end
                end
            end
        else
            % no need to combine
            res.(f) = si.(f);
        end
    end
end

function x = finalize(x,defaultop,fieldops)
for fn=fieldnames(x)'
    f = fn{1};
    if ~isempty(fieldops) && isfield(fieldops,f)
        % a field-specific op applies
        if isstruct(fieldops.(f))
            % ... which is itself a struct
            fop = fieldops.(f);
        else
            op = fieldops.(f);
        end
    else
        % the default op applies
        op = defaultop;
        fop = fieldops;
    end
    if isstruct(x.(f))
        % recurse
        x.(f) = finalize(x.(f),op,fop);
    else
        % finalize
        try
            x.(f) = op{2}(x.(f));
        catch, end
    end
end

% translate string ops into actual ops, add the default finalizer if missing
function op = translateop(op)
if isstruct(op)
    % recurse
    op = structfun(@translateop,op,'UniformOutput',false);
else
    % remap strings
    if ischar(op)
        switch op
            case 'cat'
                op = @(a,b)[a b];
            case 'replace'
                op = @(a,b)b;
            case 'sum'
                op = @(a,b)a+b;
            case 'mean'
                op = {@(a,b)[a b], @(x)mean(x)};
            case 'median'
                op = {@(a,b)[a b], @(x)median(x)};
            case 'std'
                op = {@(a,b)[a b], @(x)std(x)};
            case 'random'
                op = {@(a,b)[a b], @(x) x(min(length(x),ceil(eps+rand(1)*length(x))))};
            case 'fillblanks'
                op = @(a,b)fastif(isempty(a),b,a);
            otherwise
                error('unsupported combiner op specified');
        end
    end
    % add finalizer if missing
    if ~iscell(op)
        op = {op,@(x)x}; end
end

% inefficiently turn a struct array into a cell array of structs
function res = structarray2cellarray(arg)
res = {};
for k=1:numel(arg)
    res = [res {arg(k)}]; end
