function diff = getDirectionalDerivative(problem, x, d, storedb, key)
% Computes the directional derivative of the cost function at x along d.
%
% function diff = getDirectionalDerivative(problem, x, d)
% function diff = getDirectionalDerivative(problem, x, d, storedb)
% function diff = getDirectionalDerivative(problem, x, d, storedb, key)
%
% Returns the derivative at x along d of the cost function described in the
% problem structure.
%
% storedb is a StoreDB object, key is the StoreDB key to point x.
%
% See also: getGradient canGetDirectionalDerivative

% This file is part of Manopt: www.manopt.org.
% Original author: Nicolas Boumal, Dec. 30, 2012.
% Contributors: 
% Change log: 
%
%   April 3, 2015 (NB):
%       Works with the new StoreDB class system.

    % Allow omission of the key, and even of storedb.
    if ~exist('key', 'var')
        if ~exist('storedb', 'var')
            storedb = StoreDB();
        end
        key = storedb.getNewKey();
    end

    
    if isfield(problem, 'diff')
    %% Compute the directional derivative using diff.
        
        % Check whether this function wants to deal with storedb or not.
        switch nargin(problem.diff)
            case 2
                diff = problem.diff(x, d);
            case 3
                % Obtain, pass along, and save the store for x.
                store = storedb.getWithShared(key);
                [diff, store] = problem.diff(x, d, store);
                storedb.setWithShared(store, key);
            case 4
                % Pass along the whole storedb (by reference), with key.
                diff = problem.diff(x, d, storedb, key);
            otherwise
                up = MException('manopt:getDirectionalDerivative:baddiff', ...
                    'diff should accept 2, 3 or 4 inputs.');
                throw(up);
        end
    
    elseif canGetGradient(problem)
    %% Compute the directional derivative using the gradient.
        
        % Compute the gradient at x, then compute its inner product with d.
        grad = getGradient(problem, x, storedb, key);
        diff = problem.M.inner(x, grad, d);
        
    else
    %% Abandon computing the directional derivative.
    
        up = MException('manopt:getDirectionalDerivative:fail', ...
            ['The problem description is not explicit enough to ' ...
             'compute the directional derivatives of f.']);
        throw(up);
        
    end
    
end
