%
%       load a version of the current dataset that has been processed with Dusk2Dawn
%           - e.g. apply ASR cleaning with a particular set of parameters
%              or revert back to the raw data
%           - if no savePath is present in the data then this just loads from memory by swapping 
%               the selected version of the data (e.g. raw or cleaned with a particular set of parameters)
%               from the list of data structures found in: EEG.etc.dusk2dawn.EEG_list
% 
% Dusk2Dawn
% Author: Richard Somervail, Istituto Italiano di Tecnologia, 2022
%           www.iannettilab.net 
%%  
function EEG = d2d_loadData( varargin )


%% get inputs
EEG = varargin{1};
if nargin == 1
    cfg = [];
elseif nargin == 2
    cfg = varargin{2};
else
    error(['Error: ' mfilename ': too many input arguments'])
end

% defaults 
if ~isfield(cfg,'loadRaw'),  cfg.loadRaw  = false; end
if ~isfield(cfg,'keepList'), cfg.keepList = false; end % keep EEG_list when loading data? default is no

% get dataset filenames and paths
savePath   = EEG.etc.dusk2dawn.cfg.savePath;
origFile   = strrep(EEG.etc.dusk2dawn.cfg.origFile,'.set','');
cleanFiles = EEG.etc.dusk2dawn.cfg.cleanFiles;

if ~isempty(savePath)
    sourcestr = 'disk';
else
    sourcestr = 'memory';
end

% load either raw data or a cleaned file
if cfg.loadRaw
    fprintf('d2d_loadData: loading raw data from %s ...\n',sourcestr)
    file2load = origFile;
else % if loading cleaned data
    % find clean file
    fprintf('d2d_loadData: loading a previously-cleaned dataset from %s ...\n',sourcestr)
    file2load = d2d_getCleanedFileName(origFile,cfg);
end % if loadRaw
fprintf('d2d_loadData: dataset = %s.set ...\n', file2load )

% load dataset from file if a path is present
if ~isempty(savePath)
    EEG = pop_loadset( 'filepath', savePath,'filename', [file2load '.set'] );
else % otherwise swap the datasets from memory (EEG_list)
    % get index of cleaned file to select
    filelist  = [ {[origFile '.set']}  cleanFiles];
    old_ind   = find(strcmp(filelist,[EEG.setname '.set']));
    sel_ind   = find(strcmp(filelist,[file2load '.set']));

    % select that data and swap it out with the current one
    if ~(sel_ind == old_ind)
        EEG_list  = EEG.etc.dusk2dawn.EEG_list;
        EEG.etc.dusk2dawn = rmfield(EEG.etc.dusk2dawn,'EEG_list');
        EEG_old = EEG; % store the old data structure and remove the list of structures from it
        EEG     = EEG_list{sel_ind}; % swap the selected structure into the overall structure
        EEG_list{sel_ind} = []; % remove the selected data structure from the list of structures
        if cfg.keepList % if keeping all files in a list (not used when loading temporarily within a local scope e.g. d2d_vis_artifacts)
            EEG_list{old_ind}          = EEG_old; % store the old data structure in the list of structures
            EEG.etc.dusk2dawn.EEG_list = EEG_list; % reinsert the list of structures to the etc of the newly selected file
        end
        clear EEG_old EEG_list
    else
        if ~cfg.keepList
            % remove the list of data structures if not changing the data but not asking to keep it 
            EEG.etc.dusk2dawn = rmfield(EEG.etc.dusk2dawn,'EEG_list');
        end
    end
end

% double-check that data has requested parameter values
if ~cfg.loadRaw
    pars = EEG.etc.dusk2dawn.cfg.pars;
    npars = length(pars.values);
    if npars > 0
        checkFlag = true;
        for k = 1:npars
                thisSelection = pars.values{k}(cfg.(['sel_par_' num2str(k)]));
                thisDataset   = EEG.etc.dusk2dawn.valid(1).asr.pars.(['par_' pars.labels{k}]);
                if  ~isequal( thisSelection, thisDataset )
                    checkFlag = false;
                end
        end
    
        % if dataset doesn't match the requested parameter values then give error
        if checkFlag == false
            error( ['Error: ' mfilename ': loaded dataset doesn''t match requested dataset, submit bugreport to: www.github.com/rsomervail/dusk2dawn' ])
            %! if this ever happens then write a subfunction to return all info structs then load the dataset with matching param vals
        end
    end
end


end % END FUNCTION
