% eegplugin_fda() - EEGLAB plug-in for Functional Data Analysis (FDA).
%                   This plug-in integrates a suite of FDA methods into 
%                   EEGLAB, including:
%
%                   • Functional Principal Component Analysis (FPCA)
%                   • FPCA using the PACE algorithm (Frequency-domain FPCA)
%                   • Functional Canonical Correlation Analysis (FCCA)
%                   • Descriptive functional data statistics
%                   • Function-on-Function Regression (FoF Regression)
%
% Usage:
%   >> eegplugin_fda(fig, trystrs, catchstrs);
%
% Inputs:
%   fig        - [integer] Handle to the main EEGLAB figure.
%   trystrs    - [struct] "try" strings for EEGLAB menu callbacks.
%   catchstrs  - [struct] "catch" strings for EEGLAB menu callbacks.
%
% Outputs:
%   vers       - Plug-in version string.
%
% Version History:
%   0.1 (2021) - Initial release with core FDA functions based on the 
%                fdaM package by Ramsay, Hooker & Graves.
%
%   0.2 (2025) - Major update:
%                • Complete debugging and stability improvements using ChatGPT.
%                • Added export options for FPCA outputs to Excel.
%                • Added Function-on-Function Regression module.
%                • Added FPCA using the PACE algorithm, including support for 
%                  Frequency-Domain FPCA.
%                • Improved GUI safety, callback handling, and variable checks.
%                • Improved menu organization and enhanced EEGLAB integration.
%
% Requirements:
%   • fdaM package (included) 
%       – Only essential functions required by this plug-in are included.
%
%   • PACE FPCA toolbox (included)
%       – Only necessary files for FPCA and Frequency-Domain FPCA are included.
%
%   • Custom FDA helper modules (MyFunctions)
%
%
% Author: Mohammad Fayaz
% Contact: mohammad.fayaz.89 [at] gmail.com
%
% Reference:
%   J.O. Ramsay, G. Hooker, S. Graves (2009). 
%   "Functional Data Analysis with R and MATLAB", Springer.
%   
%   Yao F, Müller HG, Wang JL. 2005 Jun 1;100(470):577-90.
%   "Functional data analysis for sparse longitudinal data."
%   Journal of the American statistical association. 
%
% -------------------------------------------------------------------------

function vers = eegplugin_fda(fig, trystrs, catchstrs)

vers = 'FDA2.0';  % Plug-in version label

if nargin < 3
    error('eegplugin_fda requires 3 input arguments.');
end

%% ------------------------------------------------------------------------
%  Add required external toolboxes to MATLAB path
% -------------------------------------------------------------------------
addpath(genpath('fdaM'));         % Classical Functional Data Analysis
addpath(genpath('PACE'));         % PACE Functional PCA (incl. Frequency-Domain FPCA)
addpath(genpath('MyFunctions'));  % Custom FDA support functions

%% ------------------------------------------------------------------------
%  Locate EEGLAB "Tools" menu
% -------------------------------------------------------------------------
toolsmenu = findobj(fig, 'tag', 'tools');
if isempty(toolsmenu)
    warning('FDA plug-in: Could not locate EEGLAB Tools menu.');
    return;
end


%% ------------------------------------------------------------------------
%  Create callback strings for each FDA module
% -------------------------------------------------------------------------
cb2 = [ 'try, LASTCOM = pop_fda_2(EEG);' catchstrs.add_to_hist ];
cb3 = [ 'try, LASTCOM = pop_fda_3(EEG);' catchstrs.add_to_hist ];
cb4 = [ 'try, LASTCOM = pop_fda_4(EEG);' catchstrs.add_to_hist ];
cb5 = [ 'try, LASTCOM = pop_fda_5(EEG);' catchstrs.add_to_hist ];
cb6 = [ 'try, LASTCOM = pop_fda_6(EEG);' catchstrs.add_to_hist ];



%% ------------------------------------------------------------------------
%  Construct FDA menu structure in EEGLAB
% -------------------------------------------------------------------------
roi_m0  = uimenu(toolsmenu, 'label', 'FDA', 'separator', 'on');

% Epoch-based analysis
roi_m00 = uimenu(roi_m0, 'label', 'Epoch Analysis');  
uimenu(roi_m00, 'label', 'FPCA', 'CallBack', cb2);

% ERP-based and general FDA analysis
roi_m01 = uimenu(roi_m0, 'label', 'ERP Analysis');
uimenu(roi_m01, 'label', 'Descriptive FDA', 'CallBack', cb4);
uimenu(roi_m01, 'label', 'FCCA', 'CallBack', cb3);
uimenu(roi_m01, 'label', 'Function-on-Function Regression', 'CallBack', cb5);
uimenu(roi_m01, 'label', 'FPCA (PACE / Frequency-Domain)', 'CallBack', cb6);
end