% [Generalized Eigenvalue De-Artifacting Intrument (GEDAI)]
% PolyForm Noncommercial License 1.0.0
% https://polyformproject.org/licenses/noncommercial/1.0.0
%
% Copyright (C) [2025] Tomas Ros & Abele Michela
%             NeuroTuning Lab [ https://github.com/neurotuning ]
%             Center for Biomedical Imaging
%             University of Geneva
%             Switzerland
%
% For any questions, please contact:
% dr.t.ros@gmail.com

function [x, fx] = local_fminbnd(fun, ax, cx, tol)
% LOCAL_FMINBND Finds the local minimum of function in interval using Brent's method
%
% [x, fx] = local_fminbnd(fun, ax, cx, tol)
%
% Finds the local minimum of function 'fun' in interval [ax, cx] using Brent's method
% (Golden Section Search + Parabolic Interpolation).
%
% Inputs:
%   fun - Function handle to minimize
%   ax  - Lower bound of search interval
%   cx  - Upper bound of search interval
%   tol - Tolerance for convergence (default: 0.01)
%
% Outputs:
%   x   - Location of minimum
%   fx  - Function value at minimum

    if nargin < 4
        tol = 0.01;
    end
    
    % Phi is the golden ratio conjugate
    phi = (3 - sqrt(5)) / 2;
    
    % Initialize
    a = min(ax, cx);
    b = max(ax, cx);
    v = a + phi * (b - a);
    w = v;
    x = v;
    
    fv = fun(v);
    fw = fv;
    fx = fv;
    
    % Golden section step size
    d = 0.0;
    e = 0.0;
    
    iter = 0;
    max_iter = 100;
    
    while iter < max_iter
        iter = iter + 1;
        
        xm = 0.5 * (a + b);
        tol1 = tol * abs(x) + 1e-10;
        tol2 = 2.0 * tol1;
        
        % Check for convergence
        if abs(x - xm) <= (tol2 - 0.5 * (b - a))
            break;
        end
        
        if abs(e) > tol1
            % Attempt parabolic interpolation
            r = (x - w) * (fx - fv);
            q = (x - v) * (fx - fw);
            p = (x - v) * q - (x - w) * r;
            q = 2.0 * (q - r);
            if q > 0
                p = -p;
            end
            q = abs(q);
            etemp = e;
            e = d;
            
            % Check if parabolic step is acceptable
            if abs(p) >= abs(0.5 * q * etemp) || p <= q * (a - x) || p >= q * (b - x)
                % Reject parabolic, use Golden Section
                if x >= xm
                    e = a - x;
                else
                    e = b - x;
                end
                d = phi * e;
            else
                % Accept parabolic step
                d = p / q;
                u = x + d;
                if (u - a) < tol2 || (b - u) < tol2
                   d = sign(xm - x) * tol1;
                end
            end
        else
            % Golden Section step
            if x >= xm
                e = a - x;
            else
                e = b - x;
            end
            d = phi * e;
        end
        
        % Perform step
        if abs(d) >= tol1
            u = x + d;
        else
            u = x + sign(d) * tol1;
        end
        
        fu = fun(u);
        
        % Update book-keeping variables
        if fu <= fx
            if u >= x
                a = x;
            else
                b = x;
            end
            v = w; fv = fw;
            w = x; fw = fx;
            x = u; fx = fu;
        else
            if u < x
                a = u;
            else
                b = u;
            end
            if fu <= fw || w == x
                v = w; fv = fw;
                w = u; fw = fu;
            elseif fu <= fv || v == x || v == w
                v = u; fv = fu;
            end
        end
    end
end
