% pop_WearableSensing() - import WearableSensing files into EEGLAB
%
%       This function imports .CSV files produced by Wearable Sensing's 
%       DSI streaming software into EEGLAB.
%
% Usage:
%   >> OUTEEG = pop_WearableSensing; % pop up window
%   >> OUTEEG = pop_WearableSensing( filename, channels, type);
%
% Inputs:
%   filename - [string] file name
%
% Optional inputs:
%   'channels'   - [integer array] list of channel indices
%
%   'blockrange' - [min max] integer range of data blocks to import, in seconds.
%                  Entering [0 3] will import the first three blocks of data.
%                  Default is empty -> import all data blocks.
%
%  'importevent' - ['on'|'off'] import events. Default is 'on'.
%
%  'highpass'    - [real] highpass filter frequency in Hz. Default is empty.
%
%  'lowpass'     - [real] lowpass filter frequency in Hz. Default is empty.
%
%  'removeaux'   - ['on'|'off'] Remove Aux channels (X1, X2, X3). Default 'on'.
%
%  'reref'       - ['Linked Ears'|'Hardware Reference'] Reference scheme.
%
% Outputs:
%   OUTEEG   - EEGLAB data structure
%
% Author: Wearable Sensing, support@wearablesensing.com
% Copyright (C) Wearable Sensing
%
% This program is free software; you can redistribute it and/or modify
% it under the terms of the GNU General Public License as published by
% the Free Software Foundation; either version 2 of the License, or
% (at your option) any later version.
%
% This program is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU General Public License for more details.
function EEG = pop_WearableSensing(filename, varargin);
EEG = [];
Command = 0;


if nargin < 1
    % GUI mode: Prompt user to select file(s) - multiselect enabled
    [filename, filepath] = uigetfile('*.csv', 'Choose data file(s) -- pop_WearableSensing()', 'Multiselect', 'on'); 
    drawnow;
    
    if isequal(filename, 0)
        error('User cancelled');
    end
    
    % Detect batch vs single mode based on selection
    if iscell(filename)
        batchMode = true;
        numFiles = length(filename);
        filenames = filename;
    else
        batchMode = false;
        numFiles = 1;
        filenames = {filename};
    end
    
    % ===== BATCH MODE: Validate ALL files first =====
    if batchMode
        fprintf('\nBatch mode: %d files selected\n', numFiles);
        fprintf('Validating all files before processing...\n');
        
        for fileIdx = 1:numFiles
            currentFile = filenames{fileIdx};
            fullPath = [filepath currentFile];
            
            % Load file for validation
            try
                CSV_test = importdata2(fullPath);
            catch
                error('Failed to load file: %s', currentFile);
            end
            
            % Check for filtered or montage data (must be raw CSV)
            isFiltered = false;
            isMontage = false;
            
            for i = 1:length(CSV_test.textdata)
                lineLower = lower(CSV_test.textdata{i});
                
                % Check filter status
                if ~isempty(strfind(lineLower, 'filter'))
                    if isempty(strfind(lineLower, 'non-filtered')) && isempty(strfind(lineLower, 'filter_delay'))
                        isFiltered = true;
                    end
                end
                
                % Check montage status
                if ~isempty(strfind(lineLower, 'montage'))
                    isMontage = true;
                end
            end
            
            if isFiltered || isMontage
                warnMsg = sprintf('Invalid file detected in batch:\n\nFile: %s\n\nThis file appears to be filtered or montage data.\nPlease select only raw CSV files.\n\nBatch import cancelled.', currentFile);
                uiwait(warndlg(warnMsg, 'Batch Import - Invalid File'));
                error('Batch cancelled: Invalid file %s', currentFile);
            end
        end
        
        fprintf('All files validated successfully!\n\n');
        
        % Confirm batch import with user
        confirmMsg = sprintf('Batch Import Ready\n\nFiles to import: %d\n\nThe same import settings will be applied to all files.\n\nContinue?', numFiles);
        choice = questdlg(confirmMsg, 'Confirm Batch Import', 'Yes', 'No', 'Yes');
        if ~strcmp(choice, 'Yes')
            error('User cancelled batch import');
        end
    end
    
    % ===== Load FIRST file for GUI parameter setup =====
    while true
        currentFilename = filenames{1};
        fullFilename = [filepath currentFilename];
        
        % Load CSV file
        CSV = importdata2(fullFilename); 
        
        % Validate file is raw CSV (skip if already validated in batch mode)
        isFiltered = false;
        isMontage = false;
        for i = 1:length(CSV.textdata)
            lineLower = lower(CSV.textdata{i});
            
            % Check filter status
            if ~isempty(strfind(lineLower, 'filter'))
                if isempty(strfind(lineLower, 'non-filtered')) && isempty(strfind(lineLower, 'filter_delay'))
                    isFiltered = true;
                end
            end
            
            % Check montage status
            if ~isempty(strfind(lineLower, 'montage'))
                isMontage = true;
            end
        end
        
        if isFiltered || isMontage
            warnMsg = sprintf('Please make sure to upload the raw CSV file.\n\nFiltered CSV files (file names ending in _filtered) and CSV files recorded by montage (file names ending in _M) are not supported.\n\nIf needed, you can re-export the .DSI file in DSI-Streamer to obtain the raw CSV file.');
            uiwait(warndlg(warnMsg, 'Invalid File Type'));
            
            % Single file mode: prompt for new file
            if ~batchMode
                 [newFilename, newFilepath] = uigetfile('*.csv', 'Choose a data file -- pop_WearableSensing()'); 
                 drawnow;
                 if isequal(newFilename, 0)
                     error('User cancelled');
                 end
                 filenames{1} = newFilename;
                 filepath = newFilepath;
                 continue; % Restart loop with new file
            else
                 % Batch mode: Error if invalid (should be caught by pre-validation)
                 error('Invalid file encountered during batch import: %s', currentFilename);
            end
        end
        
        % File validated - proceed
        filename = fullFilename;
        break;
    end 
    
    
    
%% Specify the data to be imported from GUI
    %% Identify montage and load data from CSV
    for CheckChannel = 1:length(CSV.textdata)
        if any(strfind(CSV.textdata{CheckChannel}, 'Reference'))
            
            % --- Identify Reference Channel ---
            refLine = CSV.textdata{CheckChannel};
            parts = strsplit(refLine, {' ', ':', ',', '#', '\t'});
            parts = parts(~cellfun('isempty', parts));
            cleanParts = {};
            for k = 1:length(parts)
                word = parts{k};
                if ~strcmpi(word, 'Reference') && ...
                   ~strcmpi(word, 'location') && ...
                   ~strcmpi(word, 'Ref')
                    cleanParts{end+1} = word;
                end
            end
            if ~isempty(cleanParts)
                Reference = cleanParts{1};
            else
                Reference = 'Pz';
            end
            % ----------------------------------
            
            disp(['Reference identified as ' Reference])
            
            % --- Normalize Channel Names ---
            Reference = regexprep(Reference, '[Ss]\d+:\s*', '');
            % -------------------------------
        end
        
        % Parse channel montage (older DSI-Streamer format)
        if any(strfind(CSV.textdata{CheckChannel}, 'Sensor Position'))
            while  double(CSV.textdata{CheckChannel}(end)) == 10||double(CSV.textdata{CheckChannel}(end)) == 13
                CSV.textdata{CheckChannel}(end) = [];
            end
            tmp = regexp(CSV.textdata{CheckChannel},'([^ ,]*)','tokens');
            a = cat(1,tmp{:})';
            Montage = [a(4:end) {'Trigger'}];
            Data = CSV.data(:, 2:length(Montage)+1);
        
        % Parse channel montage (newer DSI-Streamer format)
        else
            MontageStart = strfind(CSV.colheaders, 'ime');
            MontageStart = find(not(cellfun('isempty', MontageStart)));
            MontageStart = MontageStart(1)+1;
            
            MontageEnd = strfind(CSV.colheaders, 'rigger');
            MontageEnd = find(not(cellfun('isempty', MontageEnd)));
            MontageEnd = MontageEnd(1);
            Montage = CSV.colheaders(MontageStart:MontageEnd);
            Data = CSV.data(:, MontageStart:MontageEnd);
            
        end
        
        % --- Normalize Channel Names ---
        % Strip "S#:" prefixes if present (e.g. DSI-Flex/VR300)
        Montage = regexprep(Montage, '[Ss]\d+:\s*', '');
    end
    if ~exist('Reference')  %if no reference channel specified in header, default to Pz
            Reference = 'Pz';
            disp('No reference identified.  Defaulting to Pz.')
    end
    
  % Add reference channel to montage if not already present
  if ~any(ismember(Montage, Reference))
    Montage{end + 1} = Reference;
    Data(:, end+1) = 0; 
end

    
    
    
    
    %% Identify Sampling rate (GUI input)
    
    for FreqFinder = 1:length(CSV.textdata)
        
        if any(strfind(CSV.textdata{FreqFinder}, 'Sample'))
            if any(char(regexp(CSV.textdata{FreqFinder},'\d\d\d', 'match'))) %finds 3 consecutive numerals.  
                sratepart = char(regexp(CSV.textdata{FreqFinder},'\d\d\d', 'match'));
                Srate = str2double(sratepart);
                break
            elseif any(char(regexp(CSV.textdata{FreqFinder},'\d\d', 'match'))) %finds 2 consecutive numerals, if 3 cannot be found. Probably not necessary
                sratepart = char(regexp(CSV.textdata{FreqFinder},'\d\d', 'match'));
                Srate = str2double(sratepart);
                break
            end
            
            
        end
    end
    

    
    
    % Check for data gaps caused by packet loss
    timeIdx = find(strcmpi(CSV.colheaders, 'Time'), 1);
    
    gapsFound = 0;
    totalGapTime = 0;
    boundaryEvents = [];
    
    if ~isempty(timeIdx)
        TimeVec = CSV.data(:, timeIdx);
        expectedInterval = 1 / Srate;
        actualIntervals = diff(TimeVec);
        
        % Detect gaps larger than 1.5 sample intervals
        gapIndices = find(actualIntervals > (1.5 * expectedInterval));
        
        if ~isempty(gapIndices)
            disp(' ');
            disp('--- DATA INTEGRITY CHECK ---');
            fprintf('Found %d instances of missing data.\n', length(gapIndices));
            disp('Details:');
            
            % Calculate gap statistics and record boundary locations
            for i = 1:length(gapIndices)
                idx = gapIndices(i);
                gapDuration = actualIntervals(idx);
                lostTime = gapDuration - expectedInterval;
                Ns = round(lostTime * Srate);
                
                if Ns > 0
                   timestamp = TimeVec(idx);
                   fprintf('  lost data at time %.3fs (%d samples)\n', timestamp, Ns);
                   
                   gapsFound = gapsFound + 1;
                   totalGapTime = totalGapTime + lostTime;
                   
                   % Mark boundary at start of next data segment
                   boundaryEvents = [boundaryEvents, idx + 1]; 
                end
            end
            
            disp('----------------------------');
            
            % Display summary to user (popup only for single file mode)
            if ~batchMode
                msg = sprintf('Found %d instances of missing data totaling %.2f seconds.\nSee Command Window for details.', gapsFound, totalGapTime);
                uiwait(msgbox(msg, 'Data Integrity Check'));
            end
        end
    end
    % ----------------------------------
    
    %% Identify duration of data
    [Duration, Channels] = size(Data); %size(Data) gives the number of time indices and the number of channels.
    DataDuration = (Duration/Srate); %the actual length of the data is the number of time indices divided by the sampling rate
    
    
    % Identify headset type for GUI options
    isDSI24 = false;
    for i = 1:length(CSV.textdata)
        line = CSV.textdata{i};
        if any(strfind(line, 'Headset'))
            if any(strfind(line, 'DSI-24')) || any(strfind(line, 'DSI 24'))
                isDSI24 = true;
                disp('Detected Headset: DSI-24 (Enabling Aux Removal option)');
            else
                disp(['Detected Headset: ' line]); 
            end
            break;
        end
    end

    %% Populate the GUI
    % 1. Channels (Label + Edit)
    uilist = { { 'style' 'text' 'String' 'Channels to include (default is all):' } ...
               { 'style' 'edit' 'string' '' } };
    geom = { [3 1] };
    
    % 2. Data Range (Label + Edit)
    uilist = [uilist, ...
               { { 'style' 'text' 'String' [ 'Data range (in seconds) to read (default all [0 ' int2str(DataDuration) '])' ] } }, ...
               { { 'style' 'edit' 'string' '' } } ];
    geom = [geom, { [3 1] }];
    
    % 3. Highpass Filter (Label + Edit)
    uilist = [uilist, ...
               { { 'style' 'text' 'String' 'Highpass FIR Filter (Hz) (Leave empty for no filter, recommended 0.1)' } }, ...
               { { 'style' 'edit' 'string' '0.1' } } ];
    geom = [geom, { [3 1] }];

    % 4. Lowpass Filter (Label + Edit) - Text simplified
    uilist = [uilist, ...
               { { 'style' 'text' 'String' 'Lowpass FIR Filter (Hz) (Leave empty for no filter, recommended 70)' } }, ...
               { { 'style' 'edit' 'string' '70' } } ];
    geom = [geom, { [3 1] }];
    
    % 5. Reference Scheme (Label + Popup)
    uilist = [uilist, ...
        { { 'style' 'text' 'String' 'Reference Scheme' } }, ...
        { { 'style' 'popupmenu' 'string' { 'Linked Ears', 'Hardware Reference' } 'value' 1 } } ];
    geom = [geom, { [3 1] }];

    % 6. Import Event Triggers (Label + Checkbox + Spacer) 
    uilist = [uilist, ...
        { { 'style' 'text' 'String' 'Import Event Triggers (check for "yes")' 'value' 1} }, ...
        { { 'style' 'checkbox' 'string' '' 'value' 1 } }, ...
        { {} } ];
    geom = [geom, { [3 0.35 0.5] }];
    
    % 7. Remove Aux Channels (Only if DSI-24)
    if isDSI24
        uilist = [uilist, ...
            { { 'style' 'text' 'String' 'Remove Aux Channels (X1, X2, X3)' } }, ...
            { { 'style' 'checkbox' 'string' '' 'value' 1 } }, ...
            { {} } ]; 
        geom = [geom, { [3 0.35 0.5] }];
    end
    
    GUIinput = inputgui( geom, uilist, 'pophelp(''pop_WearableSensing'')', ...
        'Load data using Wearable Sensing .csv -- pop_WearableSensing()');
    if length(GUIinput) == 0 error('User cancelled'); end; %give it nothing? it closes the GUI.
    
    % Parse GUI inputs
    options = {};
    % Standard Indices:
    % 1: Channels
    % 2: Blockrange
    % 3: Highpass
    % 4: Lowpass
    % 5: Reref
    % 6: Import Event
    
    if ~isempty(GUIinput{1}), options = { options{:} 'channels'   eval( [ '[' GUIinput{1} ']' ] ) }; end;
    if ~isempty(GUIinput{2}), options = { options{:} 'blockrange' eval( [ '[' GUIinput{2} ']' ] ) }; end;
    if ~isempty(GUIinput{3}), options = { options{:} 'highpass'   eval( [ '[' GUIinput{3} ']' ] ) }; end;
    if ~isempty(GUIinput{4}), options = { options{:} 'lowpass'    eval( [ '[' GUIinput{4} ']' ] ) }; end;
    
    reref_val = GUIinput{5};
    if ~isempty(reref_val)
        switch reref_val
            case 1
                options = { options{:} 'reref' 'Linked Ears' };
            case 2
                options = { options{:} 'reref' 'Hardware Reference' };
        end
    end
    
    if ~GUIinput{6},          options = { options{:} 'importevent'  'off'  }; end;
    
    if isDSI24
        % 7: Remove Aux
        if ~GUIinput{7}, options = { options{:} 'removeaux'  'off'  }; end;
    else
        % Default off for non-DSI-24
        options = { options{:} 'removeaux'  'off'  };
    end
    
    % ===== BATCH MODE ROUTING =====
    % If batch mode, hand off to helper function and return
    if batchMode
        % Note: Globals are managed inside process_batch_import
        process_batch_import(filepath, filenames, options);
        
        % Mark EEG as already stored to prevent duplicate in eegplugin callback
        EEG.saved = 'justloaded';
        EEG.batchimport = true;
        return; % Exit main function - batch processing complete
    end
    % ===============================
    
else
    % Command-line mode: Load file and parse vararg options
    batchMode = true;  % Suppress popups when called from process_batch_import
    options = varargin;
    CSV = importdata2(filename);
    Command = 1;
    clear EEG
   
    
    %% Identify montage (if command line input)
    for CheckChannel = 1:length(CSV.textdata)
        if any(strfind(CSV.textdata{CheckChannel}, 'Reference')) %this looks to see if in the header, a Reference channel is identified.
            
            % --- Identify Reference Channel ---
            refLine = CSV.textdata{CheckChannel};
            parts = strsplit(refLine, {' ', ':', ',', '#', '\t'});
            parts = parts(~cellfun('isempty', parts));
            cleanParts = {};
            for k = 1:length(parts)
                word = parts{k};
                if ~strcmpi(word, 'Reference') && ...
                   ~strcmpi(word, 'location') && ...
                   ~strcmpi(word, 'Ref')
                    cleanParts{end+1} = word;
                end
            end
            if ~isempty(cleanParts)
                Reference = cleanParts{1};
            else
                Reference = 'Pz';
            end
            % ----------------------------------
            
            disp(['Reference identified as ' Reference])
            
            % --- Normalize Channel Names ---
            Reference = regexprep(Reference, '[Ss]\d+:\s*', '');
            % -------------------------------
        end
        
        if any(strfind(CSV.textdata{CheckChannel}, 'Sensor Position'))
            while  double(CSV.textdata{CheckChannel}(end)) == 10||double(CSV.textdata{CheckChannel}(end)) == 13
                CSV.textdata{CheckChannel}(end) = [];
            end
            tmp = regexp(CSV.textdata{CheckChannel},'([^ ,]*)','tokens');
            a = cat(1,tmp{:})';
            Montage = [a(4:end) {'Trigger'}];
            Data = CSV.data(:, 2:length(Montage)+1);
            
        else
            MontageStart = strfind(CSV.colheaders, 'ime');
            MontageStart = find(not(cellfun('isempty', MontageStart)));
            MontageStart = MontageStart(1)+1;
            
            MontageEnd = strfind(CSV.colheaders, 'rigger');
            MontageEnd = find(not(cellfun('isempty', MontageEnd)));
            MontageEnd = MontageEnd(1);
            Montage = CSV.colheaders(MontageStart:MontageEnd);
            Data = CSV.data(:, MontageStart:MontageEnd);
            
        end
        
        % --- Normalize Channel Names ---
        Montage = regexprep(Montage, '[Ss]\d+:\s*', '');
        % -------------------------------
    end
    if ~exist('Reference')  %if no reference channel specified in header, default to Pz
            Reference = 'Pz';
            disp('No reference identified.  Defaulting to Pz.')
    end
    
  %% Insert the reference channel into the Data. (command line input)
% Update montage to include the reference channel only if it's not already present
if ~any(ismember(Montage, Reference))
    Montage{end + 1} = Reference;
    
    % --- Initialize Reference Data ---
    Data(:, end+1) = 0;
end
    
end 
  %% Identify Sampling rate (command line input)
    
    for FreqFinder = 1:length(CSV.textdata)
        
        if any(strfind(CSV.textdata{FreqFinder}, 'Sample'))
            if any(char(regexp(CSV.textdata{FreqFinder},'\d\d\d', 'match'))) %finds 3 consecutive numerals.  
                sratepart = char(regexp(CSV.textdata{FreqFinder},'\d\d\d', 'match'));
                Srate = str2double(sratepart);
                break
            elseif any(char(regexp(CSV.textdata{FreqFinder},'\d\d', 'match'))) %finds 2 consecutive numerals, if 3 cannot be found. Probably not necessary
                sratepart = char(regexp(CSV.textdata{FreqFinder},'\d\d', 'match'));
                Srate = str2double(sratepart);
                break
            end
            
            
        end
    end
    
        %% Identify duration of data (command line)
    [Duration, Channels] = size(Data); %size(Data) gives the number of time indices and the number of channels.
    DataDuration = (Duration/Srate); %the actual length of the data is the number of time indices divided by the sampling rate
    
    %% Check for data gaps (command line mode)
    timeIdx = find(strcmpi(CSV.colheaders, 'Time'), 1);
    
    gapsFound = 0;
    totalGapTime = 0;
    boundaryEvents = [];
    
    if ~isempty(timeIdx)
        TimeVec = CSV.data(:, timeIdx);
        expectedInterval = 1 / Srate;
        actualIntervals = diff(TimeVec);
        
        % Detect gaps larger than 1.5 sample intervals
        gapIndices = find(actualIntervals > (1.5 * expectedInterval));
        
        if ~isempty(gapIndices)
            disp(' ');
            disp('--- DATA INTEGRITY CHECK ---');
            fprintf('Found %d instances of missing data.\n', length(gapIndices));
            disp('Details:');
            
            % Calculate gap statistics and record boundary locations
            for i = 1:length(gapIndices)
                idx = gapIndices(i);
                gapDuration = actualIntervals(idx);
                lostTime = gapDuration - expectedInterval;
                Ns = round(lostTime * Srate);
                
                if Ns > 0
                   timestamp = TimeVec(idx);
                   fprintf('  lost data at time %.3fs (%d samples)\n', timestamp, Ns);
                   
                   gapsFound = gapsFound + 1;
                   totalGapTime = totalGapTime + lostTime;
                   
                   % Mark boundary at start of next data segment
                   boundaryEvents = [boundaryEvents, idx + 1]; 
                end
            end
            
            disp('----------------------------');
        end
    end
    
%% Make sure the input parameters make sense
%% this is where GUI and command line inputs converge

g = finputcheck( options, { 'blockrange'   'integer' [0 Inf]    [];
    'channels'     'integer' [0 Inf]    [];
    'lowpass'      'real'    [0 Srate]    [];
    'highpass'     'real'    [0 Srate]    [];
    'importevent'  'string'  { 'on';'off' } 'on';
    'removeaux'    'string'  { 'on';'off' } 'on';
    'reref'        'string'  { 'Linked Ears'; 'Hardware Reference' } 'Linked Ears';
    }, 'pop_WearableSensing');
if ischar(g), error(g); end;



%% Initialize EEGLAB set and Import Events into it

EEG = eeg_emptyset; %creates our blank slate

firststep = 0;
triggercolumn = find(ismember(Montage,'Trigger'));
if strcmp(g.importevent, 'on')
    events = [];
    for eventsearch = 1:length(Data(:,triggercolumn)) %for every index in the Trigger channel
        firststep = firststep + 1;
        if firststep == 1 %this is for the instance where the first value is nonzero
            continue
        end
        
        val = Data(eventsearch,triggercolumn);
        % Handle NaNs: Treat as 0 (Gap)
        if isnan(val)
            val = 0;
        end
        
        if val ~= 0 % if we have a nonzero trigger,
            prevVal = Data(eventsearch-1,triggercolumn);
            if isnan(prevVal), prevVal = 0; end
            
            if prevVal == 0 %and if it is preceded by zero,
                events = [events, eventsearch]; %this means it is an event.  It is added to this list.
            end
        end
    end
    if ~isempty(events)
        for event = 1:length(events)
            EEG.event(event).type = [ num2str(Data(events(event),triggercolumn)) ]; %put 'trigger ', at the beginning of the brackets to label it
            EEG.event(event).latency = events(event);
        end
    end
    
end

% --- Remove Auxiliary Channels ---
% Removes X1, X2, X3 based on their position relative to F7/F8
if strcmp(g.removeaux, 'on')
    
    f7_idx = find(strcmpi(Montage, 'F7'));
    f8_idx = find(strcmpi(Montage, 'F8'));
    
    indices_to_remove = [];
    
    % Locate X2 and X3 (Before F7)
    if ~isempty(f7_idx)
        % X2 is immediately before F7
        if f7_idx > 1
            indices_to_remove = [indices_to_remove, f7_idx - 1]; 
        end
        % X3 is 2 slots before F7
        if f7_idx > 2
            indices_to_remove = [indices_to_remove, f7_idx - 2];
        end
    end
    
    % Locate X1 (After F8)
    if ~isempty(f8_idx)
        % X1 is immediately after F8
        if f8_idx < length(Montage)
            indices_to_remove = [indices_to_remove, f8_idx + 1];
        end
    end
    
    % Remove unique indices (sorted descending)
    if ~isempty(indices_to_remove)
        indices_to_remove = sort(unique(indices_to_remove), 'descend');
        for idx = indices_to_remove
            Data(:, idx) = [];
            Montage(:, idx) = [];
        end
        disp('Removed Aux Channels');
    else
        disp('Warning: Could not locate F7/F8 anchors to remove Aux channels.');
    end
end
% ---------------------------------


%import channels

%Eliminate extraneous "channels" from data, montage

 Data(:,find(ismember(Montage,'CM'))) = [];
 Montage(:,find(ismember(Montage,'CM'))) = [];


Data(:,find(ismember(Montage,'Trigger'))) = [];
Montage(:,find(ismember(Montage,'Trigger'))) = [];

%Eliminate the channels specified in the user input from the data and montage
if ~isempty(g.channels)
    for channel = length(Montage):-1:1
        if ~any(ismember(g.channels, channel))
            Data(:,channel) = [];
            Montage(:,channel) = [];
        end
    end
end

% Limit time range if specified
if ~isempty(g.blockrange)
    interval(1) = g.blockrange(1) * Srate + 1;
    interval(2) = g.blockrange(2) * Srate;
    Data = Data(interval(1):interval(2),:);
    
    % Adjust boundary markers to match sliced data
    if ~isempty(boundaryEvents)
        validEvents = boundaryEvents(boundaryEvents >= interval(1) & boundaryEvents <= interval(2));
        boundaryEvents = validEvents - interval(1) + 1;
    end
end


% Validate filter parameters
if ~isempty(g.lowpass)
    if length(g.lowpass) ~= 1 || g.lowpass < 1
        error('Lowpass value not valid. Make sure it is a single positive value, lower than the Sampling Rate.')
    end
end
if ~isempty(g.highpass)
    if length(g.highpass) ~= 1 || g.highpass < 0
        error('Highpass value not valid. Make sure it is a single positive value.')
    end
end

% Populate EEG structure with data
EEG.srate = Srate;
EEG.data = Data';
EEG.nbchan = size(EEG.data, 1);
EEG.xmax = DataDuration;
EEG.trials = 1;
EEG.pnts = size(EEG.data,2);
EEG = eeg_checkset(EEG);

% Populate channel labels
channelindex = 1;
for label = Montage
    EEG.chanlocs(channelindex).labels = char(label);
    EEG.chanlocs(channelindex).ref = '';
    EEG.chanlocs(channelindex).theta = [];
    EEG.chanlocs(channelindex).radius= [];
    EEG.chanlocs(channelindex).X = [];
    EEG.chanlocs(channelindex).Y = [];
    EEG.chanlocs(channelindex).Z = [];
    EEG.chanlocs(channelindex).sph_theta = [];
    EEG.chanlocs(channelindex).sph_phi = [];
    EEG.chanlocs(channelindex).sph_radius = [];
    EEG.chanlocs(channelindex).type = '';
    EEG.chanlocs(channelindex).urchan = [];
    channelindex = channelindex+1;
end
EEG = eeg_checkset(EEG,'makeur');
EEG = eeg_checkset(EEG);

% Insert boundary events for data gaps
if exist('boundaryEvents', 'var') && ~isempty(boundaryEvents)
    for i = 1:length(boundaryEvents)
        EEG.event(end+1).type = 'boundary';
        EEG.event(end).latency = boundaryEvents(i);
        EEG.event(end).duration = 0;
    end
    EEG = eeg_checkset(EEG, 'eventconsistency');
end

% Apply filters if specified
if ~isempty(g.lowpass) || ~isempty(g.highpass)
	disp('Filtering data...');
	EEG = pop_eegfiltnew(EEG, g.highpass, g.lowpass);
end

% --- Post-Processing: Locations & Referencing ---
% 1. Auto-populate Channel Locations
disp('Looking up channel locations (standard_1005.elc)...');
EEG = pop_chanedit(EEG, 'lookup', 'standard_1005.elc');

% 2. Handle Re-referencing
% 2. Handle Re-referencing
if strcmp(g.reref, 'Linked Ears')
    if any(strcmpi({EEG.chanlocs.labels}, 'A1')) && any(strcmpi({EEG.chanlocs.labels}, 'A2'))
        disp('Re-referencing to Linked Ears (A1/A2 average)...');
        a1_idx = find(strcmpi({EEG.chanlocs.labels}, 'A1'));
        a2_idx = find(strcmpi({EEG.chanlocs.labels}, 'A2'));
        EEG = pop_reref(EEG, [a1_idx, a2_idx]);
        Reference = 'A1,A2';
    elseif any(strcmpi({EEG.chanlocs.labels}, 'LE'))
        disp('Re-referencing to LE channel...');
        le_idx = find(strcmpi({EEG.chanlocs.labels}, 'LE'));
        
        try
            % Manual re-referencing using local variable to avoid scope issues
            LocalStruct = EEG;
            
            % Extract ref data and subtract from all channels
            ref_data = LocalStruct.data(le_idx, :);
            LocalStruct.data = bsxfun(@minus, LocalStruct.data, ref_data);
            
            % Remove LE channel (standard behavior after re-referencing)
            LocalStruct.data(le_idx, :) = [];
            LocalStruct.chanlocs(le_idx) = [];
            
            % Update dimensions
            LocalStruct.nbchan = size(LocalStruct.data, 1);
            LocalStruct.pnts = size(LocalStruct.data, 2);
            
            % Update main EEG structure
            EEG = LocalStruct;
            EEG = eeg_checkset(EEG);
            
        catch ME
            fprintf('Error during manual re-ref: %s\n', ME.message);
            rethrow(ME);
        end
        
        Reference = 'LE'; % Update reference label
    else
        disp('Warning: Could not find A1/A2 or LE for Linked Ears referencing. Keeping original reference.');
    end
end

% 3. Update Reference Label
try
    % Use deal for safe vector assignment
    [EEG.chanlocs.ref] = deal(Reference);
catch ME
    % Fallback loop if deal fails
    for i = 1:EEG.nbchan
        EEG.chanlocs(i).ref = Reference;
    end
end

% Set dataset name from filename (without extension)
if exist('filename', 'var') && ~isempty(filename)
    [~, fname, ~] = fileparts(filename);
    EEG.setname = fname;
end

if Command == 0 %we only want EEGLAB to populate a GUI if the GUI was used
    eeglab redraw;
end

end
