function process_batch_import(filepath, filenames, options)
% PROCESS_BATCH_IMPORT - Helper function for batch importing multiple CSV files
%
% This function processes multiple WearableSensing CSV files with the same
% import settings, storing each dataset in ALLEEG.
%
% Inputs:
%   filepath  - [string] directory path containing the files
%   filenames - [cell array] list of filenames to process
%   options   - [cell array] import options (channels, filters, etc.)
%
% This function accesses global EEGLAB variables and does not return a value.

global ALLEEG EEG CURRENTSET

% Initialize global dataset counter
if isempty(CURRENTSET)
    CURRENTSET = 0;
end

numFiles = length(filenames);
fprintf('\n========================================\n');
fprintf('BATCH IMPORT: Processing %d files\n', numFiles);
fprintf('========================================\n\n');

% Initialize statistics
filesProcessed = 0;
filesWithGaps = 0;
totalGapsFound = 0;
totalGapTime = 0;
failedFiles = {};

% Iterate through selected files
for fileIdx = 1:numFiles
    currentFile = filenames{fileIdx};
    fullPath = [filepath currentFile];
    
    fprintf('[%d/%d] Processing: %s\n', fileIdx, numFiles, currentFile);
    
    try
        % Call main import function (calls pop_WearableSensing in command-line mode)
        tempEEG = pop_WearableSensing(fullPath, options{:});
        
        % Set dataset name from filename
        [~, fname, ~] = fileparts(currentFile);
        tempEEG.setname = fname;
        
        % Check for data gaps (boundary events were marked during import)
        if isfield(tempEEG, 'event') && ~isempty(tempEEG.event)
            boundaryCount = sum(strcmp({tempEEG.event.type}, 'boundary'));
            if boundaryCount > 0
                filesWithGaps = filesWithGaps + 1;
                totalGapsFound = totalGapsFound + boundaryCount;
                fprintf('  → Warning: %d gap(s) detected in this file\n', boundaryCount);
            end
        end
        
        % Store dataset in ALLEEG
        [ALLEEG, EEG, CURRENTSET] = eeg_store(ALLEEG, tempEEG, 0);
        filesProcessed = filesProcessed + 1;
        fprintf('  → Successfully imported as dataset #%d\n', CURRENTSET);
        
        % Update EEGLAB GUI
        eeglab redraw;
        
    catch ME
        % Log error but continue with remaining files
        fprintf('  → ERROR: Failed to process file\n');
        fprintf('     %s\n', ME.message);
        failedFiles{end+1} = currentFile;
    end
    
    fprintf('\n');
end

% Display summary statistics
fprintf('========================================\n');
fprintf('BATCH IMPORT COMPLETE\n');
fprintf('========================================\n');
fprintf('Files processed: %d/%d\n', filesProcessed, numFiles);

if filesWithGaps > 0
    fprintf('\nData Quality Summary:\n');
    fprintf('  Files with gaps: %d\n', filesWithGaps);
    fprintf('  Total gaps found: %d\n', totalGapsFound);
    fprintf('  (See console output above for details)\n');
end

if ~isempty(failedFiles)
    fprintf('\nFailed files (%d):\n', length(failedFiles));
    for i = 1:length(failedFiles)
        fprintf('  - %s\n', failedFiles{i});
    end
end

fprintf('========================================\n\n');

% Show completion dialog
if isempty(failedFiles)
    if filesWithGaps > 0
        summaryMsg = sprintf('Batch Import Complete\n\n%d datasets imported successfully.\n\n%d files contained data gaps.\nSee Command Window for details.', filesProcessed, filesWithGaps);
    else
        summaryMsg = sprintf('Batch Import Complete\n\n%d datasets imported successfully.\n\nNo data gaps detected.', filesProcessed);
    end
else
    summaryMsg = sprintf('Batch Import Complete\n\n%d/%d datasets imported successfully.\n\n%d files failed.\nSee Command Window for details.', filesProcessed, numFiles, length(failedFiles));
end

msgbox(summaryMsg, 'Batch Import Complete');

end
